<?php
/**
 * Uninstall Script
 *
 * Fired when the plugin is uninstalled.
 * Cleans up all plugin data including settings and converted images.
 *
 * @package Smart_Gallery_Optimizer
 */

// Exit if accessed directly or not in uninstall context
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

/**
 * Delete plugin options
 */
delete_option( 'sgo_settings' );

/**
 * Delete all galleries (custom post type)
 */
function sgo_delete_all_galleries() {
	$galleries = get_posts(
		array(
			'post_type'      => 'sgo_gallery',
			'posts_per_page' => -1,
			'post_status'    => 'any',
		)
	);

	foreach ( $galleries as $gallery ) {
		wp_delete_post( $gallery->ID, true );
	}
}

sgo_delete_all_galleries();

/**
 * Delete converted images (WebP and AVIF)
 * Note: This only deletes converted versions, original images are preserved
 */
function sgo_delete_converted_images() {
	$upload_dir = wp_upload_dir();
	$basedir = $upload_dir['basedir'];

	if ( ! is_dir( $basedir ) ) {
		return;
	}

	try {
		// Use RecursiveDirectoryIterator to find all converted images
		$directory = new RecursiveDirectoryIterator(
			$basedir,
			RecursiveDirectoryIterator::SKIP_DOTS
		);
		$iterator = new RecursiveIteratorIterator(
			$directory,
			RecursiveIteratorIterator::CHILD_FIRST
		);

		$deleted_count = 0;

		foreach ( $iterator as $file ) {
			if ( ! $file->isFile() ) {
				continue;
			}

			$extension = strtolower( $file->getExtension() );

			// Only delete WebP and AVIF files
			if ( in_array( $extension, array( 'webp', 'avif' ), true ) ) {
				$file_path = $file->getRealPath();

				// Validate the file is within uploads directory
				if ( strpos( $file_path, $basedir ) === 0 ) {
					wp_delete_file( $file_path );
					$deleted_count++;
				}
			}
		}

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( sprintf( 'SGO Uninstall: Deleted %d converted images', $deleted_count ) );
		}
	} catch ( Exception $e ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'SGO Uninstall Error: ' . $e->getMessage() );
		}
	}
}

// Execute cleanup
sgo_delete_converted_images();

// Log uninstall (only if debug mode is enabled)
if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
	error_log( 'Smart Gallery Optimizer uninstalled and cleaned up' );
}
