<?php
/**
 * Plugin Name: Smart Gallery Optimizer
 * Plugin URI: https://example.com/smart-gallery-optimizer
 * Description: Enhances gallery performance with automatic WebP/AVIF conversion, lazy loading, and SEO optimization
 * Version: 1.0.0
 * Requires at least: 5.0
 * Requires PHP: 8.0
 * Author: Sabibur Rahman
 * Author URI: https://www.fiverr.com/sabibur
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: smart-gallery-optimizer
 * Domain Path: /languages
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants
define( 'SGO_VERSION', '1.0.0' );
define( 'SGO_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SGO_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'SGO_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main Plugin Class
 */
class Smart_Gallery_Optimizer {

	/**
	 * Instance of this class
	 *
	 * @var object
	 */
	private static $instance = null;

	/**
	 * Image converter instance
	 *
	 * @var object
	 */
	public $image_converter;

	/**
	 * Admin settings instance
	 *
	 * @var object
	 */
	public $admin_settings;

	/**
	 * Shortcode handler instance
	 *
	 * @var object
	 */
	public $shortcode_handler;

	/**
	 * Gallery manager instance
	 *
	 * @var object
	 */
	public $gallery_manager;

	/**
	 * Lightbox handler instance
	 *
	 * @var object
	 */
	public $lightbox_handler;

	/**
	 * EXIF handler instance
	 *
	 * @var object
	 */
	public $exif_handler;

	/**
	 * Get singleton instance
	 *
	 * @return Smart_Gallery_Optimizer
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->load_dependencies();
		$this->init_hooks();
	}

	/**
	 * Load required files
	 */
	private function load_dependencies() {
		// Load image converter class
		require_once SGO_PLUGIN_DIR . 'includes/class-image-converter.php';

		// Load admin settings class
		require_once SGO_PLUGIN_DIR . 'includes/class-admin-settings.php';

		// Load shortcode handler class
		require_once SGO_PLUGIN_DIR . 'includes/class-shortcode-handler.php';

		// Load gallery manager class
		require_once SGO_PLUGIN_DIR . 'includes/class-gallery-manager.php';

		// Load lightbox handler class
		require_once SGO_PLUGIN_DIR . 'includes/class-lightbox-handler.php';

		// Load EXIF handler class
		require_once SGO_PLUGIN_DIR . 'includes/class-exif-handler.php';
	}

	/**
	 * Initialize WordPress hooks
	 */
	private function init_hooks() {
		// Initialize image converter
		$this->image_converter = new SGO_Image_Converter();

		// Initialize admin settings (only in admin)
		if ( is_admin() ) {
			$this->admin_settings = new SGO_Admin_Settings();
			$this->gallery_manager = new SGO_Gallery_Manager();
		}

		// Initialize shortcode handler (frontend)
		$this->shortcode_handler = new SGO_Shortcode_Handler();

		// Initialize lightbox handler (frontend)
		$this->lightbox_handler = new SGO_Lightbox_Handler();
		// Make lightbox handler globally accessible
		global $sgo_lightbox_handler;
		$sgo_lightbox_handler = $this->lightbox_handler;

		// Initialize EXIF handler
		$this->exif_handler = new SGO_EXIF_Handler();

		// Activation hook
		register_activation_hook( __FILE__, array( $this, 'activate' ) );

		// Deactivation hook
		register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
	}

	/**
	 * Plugin activation callback
	 */
	public function activate() {
		// Set default options
		$default_options = array(
			'webp_conversion'         => '1',
			'avif_conversion'         => function_exists( 'imageavif' ) ? '1' : '0',
			'webp_quality'            => 85,
			'avif_quality'            => 85,
			'lazy_loading'            => '1',
			'smart_grid_enabled'      => '1',
			'default_columns'         => 3,
			'auto_alttext_enabled'    => '1',
			'alttext_include_camera'  => '0',
			'alttext_include_date'    => '0',
			'alttext_include_location' => '0',
			'alttext_template'        => '{description} - Photo taken with {camera} on {date}',
			'default_link_type'       => 'none',
			'open_links_new_tab'      => '0',
			'default_lightbox_title'  => '0',
			'default_lightbox_caption' => '0',
		);

		add_option( 'sgo_settings', $default_options );

		// Flush rewrite rules for custom post type
		flush_rewrite_rules();

		// Log activation (only if debug mode is enabled)
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Smart Gallery Optimizer activated' );
		}
	}

	/**
	 * Plugin deactivation callback
	 */
	public function deactivate() {
		// Flush rewrite rules
		flush_rewrite_rules();

		// Log deactivation (only if debug mode is enabled)
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'Smart Gallery Optimizer deactivated' );
		}
	}

	/**
	 * Check server capabilities
	 *
	 * @return array Server capabilities
	 */
	public static function get_server_capabilities() {
		return array(
			'gd_library'    => extension_loaded( 'gd' ),
			'imagick'       => extension_loaded( 'imagick' ),
			'webp_support'  => function_exists( 'imagewebp' ),
			'avif_support'  => function_exists( 'imageavif' ),
			'exif_support'  => function_exists( 'exif_read_data' ),
		);
	}
}

/**
 * Initialize the plugin
 */
function sgo_init() {
	return Smart_Gallery_Optimizer::get_instance();
}

// Start the plugin
sgo_init();
