<?php
/**
 * Shortcode Handler Class
 *
 * Handles [smart_gallery] shortcode for responsive grid layouts
 *
 * @package Smart_Gallery_Optimizer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SGO_Shortcode_Handler Class
 */
class SGO_Shortcode_Handler {

	/**
	 * Constructor
	 */
	public function __construct() {
		// Register shortcode
		add_shortcode( 'smart_gallery', array( $this, 'render_gallery' ) );

		// Enqueue frontend assets
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	/**
	 * Render gallery shortcode
	 *
	 * @param array $atts Shortcode attributes
	 * @return string Gallery HTML
	 */
	public function render_gallery( $atts ) {
		// Get plugin settings
		$settings = get_option( 'sgo_settings', array() );
		$smart_grid_enabled = isset( $settings['smart_grid_enabled'] ) && $settings['smart_grid_enabled'] === '1';

		// If smart grid is disabled, use WordPress default gallery
		if ( ! $smart_grid_enabled ) {
			return $this->fallback_to_default_gallery( $atts );
		}

		// Check if this is a saved gallery
		$gallery_id = isset( $atts['id'] ) ? absint( $atts['id'] ) : 0;

		if ( $gallery_id > 0 ) {
			// Load gallery from database
			$gallery_data = SGO_Gallery_Manager::get_gallery_data( $gallery_id );

			if ( ! $gallery_data ) {
				return '<!-- Smart Gallery: Gallery not found -->';
			}

			// Use gallery settings with attribute overrides
			$ids     = $gallery_data['images'];
			$columns = isset( $atts['columns'] ) ? absint( $atts['columns'] ) : $gallery_data['columns'];
			$size    = isset( $atts['size'] ) ? sanitize_key( $atts['size'] ) : $gallery_data['size'];
			$gap     = isset( $atts['gap'] ) ? absint( $atts['gap'] ) : $gallery_data['gap'];
			$link_type = $gallery_data['link_type'];
			$lightbox_title = $gallery_data['lightbox_title'];
			$lightbox_caption = $gallery_data['lightbox_caption'];
		} else {
			// Legacy format with IDs attribute
			$atts = shortcode_atts(
				array(
					'ids'     => '',
					'columns' => isset( $settings['default_columns'] ) ? $settings['default_columns'] : '3',
					'size'    => 'medium',
					'gap'     => '20',
				),
				$atts,
				'smart_gallery'
			);

			// Validate and sanitize attributes
			$ids     = $this->parse_ids( $atts['ids'] );
			$columns = $this->validate_columns( $atts['columns'] );
			$size    = $this->validate_size( $atts['size'] );
			$gap     = absint( $atts['gap'] );

			// Get default link settings
			$link_type = isset( $settings['default_link_type'] ) ? $settings['default_link_type'] : 'none';
			$lightbox_title = isset( $settings['default_lightbox_title'] ) && $settings['default_lightbox_title'];
			$lightbox_caption = isset( $settings['default_lightbox_caption'] ) && $settings['default_lightbox_caption'];
		}

		// Return empty if no valid IDs
		if ( empty( $ids ) || ! is_array( $ids ) ) {
			return '';
		}

		// Get lazy loading setting
		$lazy_loading = isset( $settings['lazy_loading'] ) && $settings['lazy_loading'] === '1';

		// Generate gallery HTML
		return $this->generate_gallery_html(
			$ids,
			$columns,
			$size,
			$gap,
			$lazy_loading,
			$link_type,
			$gallery_id ? 'gallery-' . $gallery_id : 'gallery-' . wp_rand(),
			$lightbox_title,
			$lightbox_caption
		);
	}

	/**
	 * Parse and validate image IDs
	 *
	 * @param string $ids_string Comma-separated IDs
	 * @return array Valid attachment IDs
	 */
	private function parse_ids( $ids_string ) {
		if ( empty( $ids_string ) ) {
			return array();
		}

		// Split by comma and sanitize
		$ids = explode( ',', $ids_string );
		$ids = array_map( 'trim', $ids );
		$ids = array_map( 'absint', $ids );

		// Remove zeros and duplicates
		$ids = array_filter( $ids );
		$ids = array_unique( $ids );

		// Verify attachments exist
		$valid_ids = array();
		foreach ( $ids as $id ) {
			if ( wp_attachment_is_image( $id ) ) {
				$valid_ids[] = $id;
			}
		}

		return $valid_ids;
	}

	/**
	 * Validate columns attribute
	 *
	 * @param mixed $columns Columns value
	 * @return int Valid columns (2-6)
	 */
	private function validate_columns( $columns ) {
		$columns = absint( $columns );

		// Ensure within valid range
		if ( $columns < 2 ) {
			$columns = 2;
		} elseif ( $columns > 6 ) {
			$columns = 6;
		}

		return $columns;
	}

	/**
	 * Validate size attribute
	 *
	 * @param string $size Image size
	 * @return string Valid image size
	 */
	private function validate_size( $size ) {
		$size = sanitize_key( $size );

		// Get all registered image sizes
		$valid_sizes = array_merge(
			array( 'thumbnail', 'medium', 'medium_large', 'large', 'full' ),
			array_keys( wp_get_additional_image_sizes() )
		);

		// Default to medium if invalid
		if ( ! in_array( $size, $valid_sizes, true ) ) {
			$size = 'medium';
		}

		return $size;
	}

	/**
	 * Generate gallery HTML
	 *
	 * @param array  $ids Image IDs
	 * @param int    $columns Number of columns
	 * @param string $size Image size
	 * @param int    $gap Gap between images
	 * @param bool   $lazy_loading Enable lazy loading
	 * @param string $link_type Link type (none/direct/lightbox)
	 * @param string $gallery_id Unique gallery ID
	 * @param bool   $lightbox_title Show title in lightbox
	 * @param bool   $lightbox_caption Show caption in lightbox
	 * @return string Gallery HTML
	 */
	private function generate_gallery_html( $ids, $columns, $size, $gap, $lazy_loading, $link_type = 'none', $gallery_id = '', $lightbox_title = false, $lightbox_caption = false ) {
		$output = '';

		// Start gallery container
		$output .= sprintf(
			'<div class="sgo-gallery" data-columns="%d" data-gap="%d" style="--sgo-gap: %dpx;">',
			esc_attr( $columns ),
			esc_attr( $gap ),
			esc_attr( $gap )
		);

		// Loop through images
		foreach ( $ids as $id ) {
			$output .= $this->render_gallery_item(
				$id,
				$size,
				$lazy_loading,
				$link_type,
				$gallery_id,
				$lightbox_title,
				$lightbox_caption
			);
		}

		// Close gallery container
		$output .= '</div>';

		return $output;
	}

	/**
	 * Render single gallery item
	 *
	 * @param int    $attachment_id Attachment ID
	 * @param string $size Image size
	 * @param bool   $lazy_loading Enable lazy loading
	 * @param string $link_type Link type (none/direct/lightbox)
	 * @param string $gallery_id Unique gallery ID
	 * @param bool   $lightbox_title Show title in lightbox
	 * @param bool   $lightbox_caption Show caption in lightbox
	 * @return string Gallery item HTML
	 */
	private function render_gallery_item( $attachment_id, $size, $lazy_loading, $link_type = 'none', $gallery_id = '', $lightbox_title = false, $lightbox_caption = false ) {
		// Get image data
		$image_src = wp_get_attachment_image_src( $attachment_id, $size );
		$image_srcset = wp_get_attachment_image_srcset( $attachment_id, $size );
		$image_alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		$image_title = get_the_title( $attachment_id );

		if ( ! $image_src ) {
			return '';
		}

		$image_url = $image_src[0];

		// Check if WebP version exists
		$webp_url = $this->get_converted_image_url( $attachment_id, $size, 'webp' );
		if ( $webp_url ) {
			$image_url = $webp_url;
		}

		// Start gallery item
		$output = '<figure class="sgo-gallery-item">';

		// Generate image tag
		if ( $lazy_loading ) {
			// Lazy loading enabled
			$image_html = sprintf(
				'<img class="sgo-lazy" src="%s" data-src="%s" %s alt="%s" title="%s">',
				esc_url( $this->get_placeholder_image() ),
				esc_url( $image_url ),
				$image_srcset ? 'data-srcset="' . esc_attr( $image_srcset ) . '"' : '',
				esc_attr( $image_alt ),
				esc_attr( $image_title )
			);
		} else {
			// Lazy loading disabled
			$image_html = sprintf(
				'<img src="%s" %s alt="%s" title="%s">',
				esc_url( $image_url ),
				$image_srcset ? 'srcset="' . esc_attr( $image_srcset ) . '"' : '',
				esc_attr( $image_alt ),
				esc_attr( $image_title )
			);
		}

		// Get lightbox handler
		global $sgo_lightbox_handler;

		// Process link behavior
		if ( $link_type && $link_type !== 'none' && $sgo_lightbox_handler ) {
			$output .= $sgo_lightbox_handler->process_image_link(
				$attachment_id,
				$image_html,
				$link_type,
				$gallery_id,
				$lightbox_title,
				$lightbox_caption
			);
		} else {
			$output .= $image_html;
		}

		// Close gallery item
		$output .= '</figure>';

		return $output;
	}

	/**
	 * Get converted image URL (WebP/AVIF)
	 *
	 * @param int    $attachment_id Attachment ID
	 * @param string $size Image size
	 * @param string $format Image format (webp/avif)
	 * @return string|false Converted image URL or false
	 */
	private function get_converted_image_url( $attachment_id, $size, $format = 'webp' ) {
		// Check if image was converted
		$converted = get_post_meta( $attachment_id, '_sgo_converted', true );
		if ( ! $converted ) {
			return false;
		}

		// Get original image path
		$image_src = wp_get_attachment_image_src( $attachment_id, $size );
		if ( ! $image_src ) {
			return false;
		}

		$image_url = $image_src[0];

		// Get upload directory info
		$upload_dir = wp_upload_dir();

		// Convert URL to path
		$image_path = str_replace( $upload_dir['baseurl'], $upload_dir['basedir'], $image_url );

		// Build converted image path
		$file_info = pathinfo( $image_path );
		$converted_path = $file_info['dirname'] . '/' . $file_info['filename'] . '.' . $format;

		// Check if converted file exists
		if ( ! file_exists( $converted_path ) ) {
			return false;
		}

		// Convert path back to URL
		$converted_url = str_replace( $upload_dir['basedir'], $upload_dir['baseurl'], $converted_path );

		return $converted_url;
	}

	/**
	 * Get placeholder image (1x1 transparent pixel)
	 *
	 * @return string Data URI of placeholder
	 */
	private function get_placeholder_image() {
		// 1x1 transparent GIF
		return 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';
	}

	/**
	 * Fallback to WordPress default gallery
	 *
	 * @param array $atts Shortcode attributes
	 * @return string Gallery HTML
	 */
	private function fallback_to_default_gallery( $atts ) {
		// Map attributes to WordPress gallery shortcode
		$gallery_atts = array();

		if ( isset( $atts['ids'] ) ) {
			$gallery_atts['ids'] = $atts['ids'];
		}

		if ( isset( $atts['columns'] ) ) {
			$gallery_atts['columns'] = $atts['columns'];
		}

		if ( isset( $atts['size'] ) ) {
			$gallery_atts['size'] = $atts['size'];
		}

		// Use WordPress core gallery_shortcode function
		return gallery_shortcode( $gallery_atts );
	}

	/**
	 * Enqueue frontend assets
	 */
	public function enqueue_assets() {
		global $post;

		// Only enqueue if shortcode is present
		if ( ! is_a( $post, 'WP_Post' ) || ! has_shortcode( $post->post_content, 'smart_gallery' ) ) {
			return;
		}

		// Get plugin settings
		$settings = get_option( 'sgo_settings', array() );
		$lazy_loading = isset( $settings['lazy_loading'] ) && $settings['lazy_loading'] === '1';
		$smart_grid_enabled = isset( $settings['smart_grid_enabled'] ) && $settings['smart_grid_enabled'] === '1';

		// Only load assets if features are enabled
		if ( ! $smart_grid_enabled && ! $lazy_loading ) {
			return;
		}

		// Determine if we should use minified files
		$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		// Enqueue CSS
		if ( $smart_grid_enabled ) {
			wp_enqueue_style(
				'sgo-frontend-style',
				SGO_PLUGIN_URL . 'assets/css/frontend-style' . $suffix . '.css',
				array(),
				SGO_VERSION,
				'all'
			);
		}

		// Enqueue JS for lazy loading
		if ( $lazy_loading ) {
			wp_enqueue_script(
				'sgo-lazy-load',
				SGO_PLUGIN_URL . 'assets/js/lazy-load' . $suffix . '.js',
				array(),
				SGO_VERSION,
				array(
					'in_footer' => true,
					'strategy'  => 'defer',
				)
			);
		}
	}
}
