<?php
/**
 * Lightbox Handler Class
 *
 * Handles lightbox integration using PhotoSwipe library
 *
 * @package Smart_Gallery_Optimizer
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class SGO_Lightbox_Handler
 *
 * Manages lightbox functionality for galleries using PhotoSwipe
 */
class SGO_Lightbox_Handler {

	/**
	 * Track if lightbox is needed on current page
	 *
	 * @var bool
	 */
	private $lightbox_needed = false;

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'wp_enqueue_scripts', array( $this, 'maybe_enqueue_lightbox' ), 20 );
	}

	/**
	 * Mark that lightbox is needed on this page
	 */
	public function set_lightbox_needed() {
		$this->lightbox_needed = true;
	}

	/**
	 * Check if lightbox is needed
	 *
	 * @return bool
	 */
	public function is_lightbox_needed() {
		return $this->lightbox_needed;
	}

	/**
	 * Conditionally enqueue lightbox assets
	 */
	public function maybe_enqueue_lightbox() {
		global $post;

		// Check if current post has smart_gallery shortcode
		if ( ! is_a( $post, 'WP_Post' ) || ! has_shortcode( $post->post_content, 'smart_gallery' ) ) {
			return;
		}

		// Check if any saved galleries exist with lightbox enabled
		$has_lightbox = false;

		// Parse shortcodes to find gallery IDs
		if ( preg_match_all( '/\[smart_gallery[^\]]*id=["\']?(\d+)["\']?[^\]]*\]/i', $post->post_content, $matches ) ) {
			foreach ( $matches[1] as $gallery_id ) {
				$gallery_data = SGO_Gallery_Manager::get_gallery_data( absint( $gallery_id ) );
				if ( $gallery_data && $gallery_data['link_type'] === 'lightbox' ) {
					$has_lightbox = true;
					break;
				}
			}
		}

		// Also check if legacy shortcode with link settings
		if ( ! $has_lightbox ) {
			$settings = get_option( 'sgo_settings', array() );
			$default_link_type = isset( $settings['default_link_type'] ) ? $settings['default_link_type'] : 'none';

			// If default is lightbox and there's a gallery on the page, enable it
			if ( $default_link_type === 'lightbox' ) {
				$has_lightbox = true;
			}
		}

		// Only enqueue if lightbox is needed
		if ( ! $has_lightbox ) {
			return;
		}

		// Enqueue PhotoSwipe CSS (from local photoswipe folder)
		wp_enqueue_style(
			'photoswipe',
			SGO_PLUGIN_URL . 'assets/photoswipe/photoswipe.min.css',
			array(),
			'5.4.4'
		);

		// Enqueue custom caption CSS
		wp_enqueue_style(
			'sgo-lightbox-caption',
			SGO_PLUGIN_URL . 'assets/css/lightbox-caption.css',
			array( 'photoswipe' ),
			SGO_VERSION
		);

		// Enqueue PhotoSwipe core JS (from local photoswipe folder)
		wp_enqueue_script(
			'photoswipe',
			SGO_PLUGIN_URL . 'assets/photoswipe/photoswipe.umd.min.js',
			array(),
			'5.4.4',
			true
		);

		// Enqueue PhotoSwipe Lightbox JS (from local photoswipe folder)
		wp_enqueue_script(
			'photoswipe-lightbox',
			SGO_PLUGIN_URL . 'assets/photoswipe/photoswipe-lightbox.umd.min.js',
			array( 'photoswipe' ),
			'5.4.4',
			true
		);

		// Enqueue our custom lightbox initialization script
		wp_enqueue_script(
			'sgo-lightbox-init',
			SGO_PLUGIN_URL . 'assets/js/lightbox-init.js',
			array( 'photoswipe', 'photoswipe-lightbox' ),
			SGO_VERSION,
			true
		);

		// Pass configuration to JavaScript
		wp_localize_script(
			'sgo-lightbox-init',
			'sgoLightbox',
			array(
				'gallerySelector'     => '.sgo-gallery',
				'childSelector'       => 'a.sgo-lightbox',
				'showHideAnimationType' => 'zoom',
				'zoomAnimationDuration' => 333,
				'bgOpacity'           => 0.9,
				'spacing'             => 0.1,
				'allowPanToNext'      => true,
				'loop'                => true,
				'pinchToClose'        => true,
				'closeOnVerticalDrag' => true,
				'escKey'              => true,
				'arrowKeys'           => true,
				'returnFocus'         => true,
				'clickToCloseNonZoomable' => true,
				'imageClickAction'    => 'zoom',
				'tapAction'           => 'toggle-controls',
				'doubleTapAction'     => 'zoom',
				'preloadFirstSlide'   => true,
				'showAnimationDuration' => 333,
				'hideAnimationDuration' => 333,
			)
		);
	}

	/**
	 * Generate lightbox link HTML for PhotoSwipe
	 *
	 * @param int    $image_id        Image attachment ID
	 * @param string $image_html      Image HTML
	 * @param string $gallery_id      Gallery ID for grouping
	 * @param bool   $show_title      Show title in lightbox
	 * @param bool   $show_caption    Show caption in lightbox
	 * @return string Lightbox link HTML
	 */
	public function generate_lightbox_link( $image_id, $image_html, $gallery_id, $show_title = false, $show_caption = false ) {
		// Get full-size image URL and dimensions
		$full_image = wp_get_attachment_image_src( $image_id, 'full' );

		if ( ! $full_image ) {
			return $image_html;
		}

		$full_url = $full_image[0];
		$width = absint( $full_image[1] );
		$height = absint( $full_image[2] );

		// Fallback dimensions if not available
		if ( $width === 0 || $height === 0 ) {
			$metadata = wp_get_attachment_metadata( $image_id );
			if ( ! empty( $metadata['width'] ) && ! empty( $metadata['height'] ) ) {
				$width = absint( $metadata['width'] );
				$height = absint( $metadata['height'] );
			} else {
				// Default fallback dimensions
				$width = 1200;
				$height = 900;
			}
		}

		// Get image title and caption
		$title = '';
		$caption = '';

		if ( $show_title ) {
			$title = get_the_title( $image_id );
		}

		if ( $show_caption ) {
			$attachment = get_post( $image_id );
			if ( $attachment && ! empty( $attachment->post_excerpt ) ) {
				$caption = $attachment->post_excerpt;
			}
		}

		// Build caption HTML (combining title and description)
		$caption_html = '';
		if ( $title || $caption ) {
			if ( $title && $caption ) {
				$caption_html = '<strong>' . esc_html( $title ) . '</strong><br>' . esc_html( $caption );
			} elseif ( $title ) {
				$caption_html = '<strong>' . esc_html( $title ) . '</strong>';
			} else {
				$caption_html = esc_html( $caption );
			}
		}

		// Build data attributes for PhotoSwipe
		$data_attrs = array();
		$data_attrs[] = 'data-pswp-src="' . esc_url( $full_url ) . '"';
		$data_attrs[] = 'data-pswp-width="' . esc_attr( $width ) . '"';
		$data_attrs[] = 'data-pswp-height="' . esc_attr( $height ) . '"';

		if ( $caption_html ) {
			$data_attrs[] = 'data-pswp-caption="' . esc_attr( $caption_html ) . '"';
		}

		// Add data attribute to prevent theme lightbox from interfering
		$data_attrs[] = 'data-no-swipebox="true"';

		// Build lightbox link (removed target="_blank" to allow PhotoSwipe to handle click)
		$link_html = sprintf(
			'<a href="%s" class="sgo-lightbox" %s>%s</a>',
			esc_url( $full_url ),
			implode( ' ', $data_attrs ),
			$image_html
		);

		return $link_html;
	}

	/**
	 * Generate direct link HTML
	 *
	 * @param int    $image_id   Image attachment ID
	 * @param string $image_html Image HTML
	 * @return string Direct link HTML
	 */
	public function generate_direct_link( $image_id, $image_html ) {
		// Get full-size image URL
		$full_url = wp_get_attachment_image_url( $image_id, 'full' );

		if ( ! $full_url ) {
			return $image_html;
		}

		// Get settings for target behavior
		$settings = get_option( 'sgo_settings', array() );
		$new_tab = isset( $settings['open_links_new_tab'] ) && $settings['open_links_new_tab'];

		$target = $new_tab ? ' target="_blank" rel="noopener noreferrer"' : '';

		// Build direct link
		$link_html = sprintf(
			'<a href="%s" class="sgo-direct-link"%s>%s</a>',
			esc_url( $full_url ),
			$target,
			$image_html
		);

		return $link_html;
	}

	/**
	 * Process image link based on type
	 *
	 * @param int    $image_id     Image attachment ID
	 * @param string $image_html   Image HTML
	 * @param string $link_type    Link type: 'none', 'direct', 'lightbox'
	 * @param string $gallery_id   Gallery ID for grouping
	 * @param bool   $show_title   Show title in lightbox
	 * @param bool   $show_caption Show caption in lightbox
	 * @return string Final image HTML (with or without link)
	 */
	public function process_image_link( $image_id, $image_html, $link_type, $gallery_id, $show_title = false, $show_caption = false ) {
		switch ( $link_type ) {
			case 'direct':
				return $this->generate_direct_link( $image_id, $image_html );

			case 'lightbox':
				$this->set_lightbox_needed();
				return $this->generate_lightbox_link( $image_id, $image_html, $gallery_id, $show_title, $show_caption );

			case 'none':
			default:
				return $image_html;
		}
	}
}
