<?php
/**
 * Image Converter Class
 *
 * Handles automatic conversion of uploaded images to WebP and AVIF formats
 *
 * @package Smart_Gallery_Optimizer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SGO_Image_Converter Class
 */
class SGO_Image_Converter {

	/**
	 * Log debug message (only when WP_DEBUG is enabled)
	 *
	 * @param string $message Message to log
	 */
	private function log( $message ) {
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'SGO: ' . $message );
		}
	}

	/**
	 * Constructor
	 */
	public function __construct() {
		// Hook into image metadata generation (after all sizes are created)
		add_filter( 'wp_generate_attachment_metadata', array( $this, 'convert_all_image_sizes' ), 10, 2 );

		// Hook into image tag generation for lazy loading
		add_filter( 'wp_get_attachment_image_attributes', array( $this, 'add_lazy_loading_attributes' ), 10, 3 );

		// Hook into image tag generation for fallback
		add_filter( 'wp_get_attachment_image_attributes', array( $this, 'add_srcset_attributes' ), 10, 3 );
		add_filter( 'wp_calculate_image_srcset', array( $this, 'add_webp_to_srcset' ), 10, 5 );
	}

	/**
	 * Convert all generated image sizes to WebP and AVIF
	 *
	 * @param array $metadata Attachment metadata
	 * @param int   $attachment_id Attachment ID
	 * @return array Modified metadata
	 */
	public function convert_all_image_sizes( $metadata, $attachment_id ) {
		// Skip during bulk operations
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return $metadata;
		}

		// Get plugin settings
		$settings = get_option( 'sgo_settings', array() );
		$webp_enabled = isset( $settings['webp_conversion'] ) && $settings['webp_conversion'] === '1';
		$avif_enabled = isset( $settings['avif_conversion'] ) && $settings['avif_conversion'] === '1';

		// Skip if both are disabled
		if ( ! $webp_enabled && ! $avif_enabled ) {
			return $metadata;
		}

		// Get original file path
		$original_file = get_attached_file( $attachment_id );
		if ( ! $original_file || ! file_exists( $original_file ) ) {
			return $metadata;
		}

		// Check if this is an image
		$file_type = wp_check_filetype( $original_file );
		if ( strpos( $file_type['type'], 'image/' ) !== 0 ) {
			return $metadata;
		}

		$file_info = pathinfo( $original_file );
		$extension = strtolower( $file_info['extension'] );

		// Only convert JPEG and PNG
		if ( ! in_array( $extension, array( 'jpg', 'jpeg', 'png' ), true ) ) {
			return $metadata;
		}

		$upload_dir = wp_upload_dir();
		$conversion_status = array();

		// Convert original image
		$conversion_status['original'] = $this->convert_image_formats( $original_file, $extension, $webp_enabled, $avif_enabled );

		// Convert all generated sizes
		if ( isset( $metadata['sizes'] ) && is_array( $metadata['sizes'] ) ) {
			foreach ( $metadata['sizes'] as $size_name => $size_data ) {
				$size_file = path_join( dirname( $original_file ), $size_data['file'] );

				if ( file_exists( $size_file ) ) {
					$conversion_status[ $size_name ] = $this->convert_image_formats( $size_file, $extension, $webp_enabled, $avif_enabled );
				}
			}
		}

		// Store conversion status in post meta
		update_post_meta( $attachment_id, '_sgo_conversion_status', $conversion_status );
		update_post_meta( $attachment_id, '_sgo_converted', true );

		return $metadata;
	}

	/**
	 * Convert single image to WebP and/or AVIF
	 *
	 * @param string $source_path Source image path
	 * @param string $extension Source image extension
	 * @param bool   $webp_enabled Whether to convert to WebP
	 * @param bool   $avif_enabled Whether to convert to AVIF
	 * @return array Conversion results
	 */
	private function convert_image_formats( $source_path, $extension, $webp_enabled, $avif_enabled ) {
		$results = array(
			'webp' => false,
			'avif' => false,
		);

		// Check file size before conversion
		if ( ! file_exists( $source_path ) ) {
			$this->log( 'SGO: Source file does not exist - ' . $source_path );
			return $results;
		}

		$file_size = filesize( $source_path );

		// Skip very large files (>50MB) to prevent server overload
		if ( $file_size > 50 * 1024 * 1024 ) {
			$this->log( 'SGO: File too large for conversion - ' . $source_path . ' (' . size_format( $file_size ) . ')' );
			return $results;
		}

		// Skip very small files (<1KB) - likely corrupted or placeholder
		if ( $file_size < 1024 ) {
			$this->log( 'SGO: File too small for conversion - ' . $source_path );
			return $results;
		}

		$file_info = pathinfo( $source_path );

		// Convert to WebP
		if ( $webp_enabled && function_exists( 'imagewebp' ) ) {
			$webp_path = $file_info['dirname'] . '/' . $file_info['filename'] . '.webp';
			$results['webp'] = $this->create_webp_image( $source_path, $webp_path, $extension );
		}

		// Convert to AVIF
		if ( $avif_enabled && function_exists( 'imageavif' ) ) {
			$avif_path = $file_info['dirname'] . '/' . $file_info['filename'] . '.avif';
			$results['avif'] = $this->create_avif_image( $source_path, $avif_path, $extension );
		}

		return $results;
	}

	/**
	 * Create WebP image from source
	 *
	 * @param string $source_path Source image path
	 * @param string $dest_path Destination WebP path
	 * @param string $extension Source image extension
	 * @return bool Success status
	 */
	private function create_webp_image( $source_path, $dest_path, $extension ) {
		// Validate source file
		if ( ! file_exists( $source_path ) ) {
			$this->log( 'SGO: Source file not found - ' . $source_path );
			return false;
		}

		if ( ! is_readable( $source_path ) ) {
			$this->log( 'SGO: Cannot read source file - ' . $source_path );
			return false;
		}

		// Check destination is writable
		$dest_dir = dirname( $dest_path );
		if ( ! is_writable( $dest_dir ) ) {
			$this->log( 'SGO: Destination directory not writable - ' . $dest_dir );
			return false;
		}

		// Manage memory for large files
		$this->manage_memory_for_image( $source_path );

		try {
			// Load source image
			$image = $this->load_image( $source_path, $extension );

			if ( ! $image ) {
				$this->log( 'SGO: Failed to load source image - ' . $source_path );
				return false;
			}

			// Get quality setting
			$settings = get_option( 'sgo_settings', array() );
			$quality = isset( $settings['webp_quality'] ) ? intval( $settings['webp_quality'] ) : 85;

			// Convert to WebP
			$result = @imagewebp( $image, $dest_path, $quality );

			// Free memory
			imagedestroy( $image );

			if ( $result ) {
				$this->log( 'SGO: WebP created - ' . $dest_path );
			} else {
				$this->log( 'SGO: Failed to create WebP - ' . $dest_path );
			}

			return $result;

		} catch ( Exception $e ) {
			$this->log( 'SGO: Exception creating WebP - ' . $e->getMessage() );
			return false;
		}
	}

	/**
	 * Create AVIF image from source
	 *
	 * @param string $source_path Source image path
	 * @param string $dest_path Destination AVIF path
	 * @param string $extension Source image extension
	 * @return bool Success status
	 */
	private function create_avif_image( $source_path, $dest_path, $extension ) {
		// Check if AVIF is supported
		if ( ! function_exists( 'imageavif' ) ) {
			return false;
		}

		// Validate source file
		if ( ! file_exists( $source_path ) ) {
			$this->log( 'SGO: Source file not found - ' . $source_path );
			return false;
		}

		if ( ! is_readable( $source_path ) ) {
			$this->log( 'SGO: Cannot read source file - ' . $source_path );
			return false;
		}

		// Check destination is writable
		$dest_dir = dirname( $dest_path );
		if ( ! is_writable( $dest_dir ) ) {
			$this->log( 'SGO: Destination directory not writable - ' . $dest_dir );
			return false;
		}

		// Manage memory for large files
		$this->manage_memory_for_image( $source_path );

		try {
			// Load source image
			$image = $this->load_image( $source_path, $extension );

			if ( ! $image ) {
				$this->log( 'SGO: Failed to load source image - ' . $source_path );
				return false;
			}

			// Get quality setting
			$settings = get_option( 'sgo_settings', array() );
			$quality = isset( $settings['avif_quality'] ) ? intval( $settings['avif_quality'] ) : 85;

			// Convert to AVIF
			$result = @imageavif( $image, $dest_path, $quality );

			// Free memory
			imagedestroy( $image );

			if ( $result ) {
				$this->log( 'SGO: AVIF created - ' . $dest_path );
			} else {
				$this->log( 'SGO: Failed to create AVIF - ' . $dest_path );
			}

			return $result;

		} catch ( Exception $e ) {
			$this->log( 'SGO: Exception creating AVIF - ' . $e->getMessage() );
			return false;
		}
	}

	/**
	 * Load image resource from file
	 *
	 * @param string $file_path Image file path
	 * @param string $extension Image extension
	 * @return resource|false Image resource or false on failure
	 */
	private function load_image( $file_path, $extension ) {
		$image = false;

		try {
			switch ( $extension ) {
				case 'jpg':
				case 'jpeg':
					$image = @imagecreatefromjpeg( $file_path );
					break;
				case 'png':
					$image = @imagecreatefrompng( $file_path );
					// Preserve transparency
					if ( $image ) {
						imagealphablending( $image, true );
						imagesavealpha( $image, true );
					}
					break;
			}
		} catch ( Exception $e ) {
			$this->log( 'SGO: Exception loading image - ' . $e->getMessage() );
			return false;
		}

		return $image;
	}

	/**
	 * Manage memory allocation for image processing
	 *
	 * @param string $file_path Image file path
	 */
	private function manage_memory_for_image( $file_path ) {
		$file_size = filesize( $file_path );

		// For files larger than 5MB, increase memory limit
		if ( $file_size > 5 * 1024 * 1024 ) {
			$current_limit = ini_get( 'memory_limit' );
			$current_limit_mb = intval( $current_limit );

			// Set to at least 256MB for large images
			if ( $current_limit_mb < 256 ) {
				@ini_set( 'memory_limit', '256M' );
			}
		}
	}

	/**
	 * Add WebP/AVIF sources to srcset
	 *
	 * @param array  $sources Array of image sources
	 * @param array  $size_array Image size array
	 * @param string $image_src Image source URL
	 * @param array  $image_meta Image metadata
	 * @param int    $attachment_id Attachment ID
	 * @return array Modified sources
	 */
	public function add_webp_to_srcset( $sources, $size_array, $image_src, $image_meta, $attachment_id ) {
		// Get plugin settings
		$settings = get_option( 'sgo_settings', array() );
		$webp_enabled = isset( $settings['webp_conversion'] ) && $settings['webp_conversion'] === '1';

		if ( ! $webp_enabled ) {
			return $sources;
		}

		// Check if image was converted
		$converted = get_post_meta( $attachment_id, '_sgo_converted', true );
		if ( ! $converted ) {
			return $sources;
		}

		// Get upload directory info
		$upload_dir = wp_upload_dir();

		// Add WebP versions to srcset
		foreach ( $sources as $width => $source ) {
			$file_info = pathinfo( $source['url'] );
			$webp_url = $file_info['dirname'] . '/' . $file_info['filename'] . '.webp';
			$webp_path = str_replace( $upload_dir['baseurl'], $upload_dir['basedir'], $webp_url );

			// Validate that the path is within the uploads directory
			$real_webp_path = realpath( $webp_path );
			$real_upload_dir = realpath( $upload_dir['basedir'] );

			if ( $real_webp_path && $real_upload_dir &&
			     strpos( $real_webp_path, $real_upload_dir ) === 0 &&
			     file_exists( $real_webp_path ) ) {
				$sources[ $width ]['url'] = $webp_url;
			}
		}

		return $sources;
	}

	/**
	 * Add lazy loading attributes to images
	 *
	 * @param array  $attr Image attributes
	 * @param object $attachment Attachment object
	 * @param string $size Image size
	 * @return array Modified attributes
	 */
	public function add_lazy_loading_attributes( $attr, $attachment, $size ) {
		// Get plugin settings
		$settings = get_option( 'sgo_settings', array() );
		$lazy_loading = isset( $settings['lazy_loading'] ) && $settings['lazy_loading'] === '1';

		// Skip if lazy loading is disabled
		if ( ! $lazy_loading ) {
			return $attr;
		}

		// Skip in admin
		if ( is_admin() ) {
			return $attr;
		}

		// Add lazy loading attributes
		if ( isset( $attr['src'] ) ) {
			// Store original src in data-src
			$attr['data-src'] = $attr['src'];

			// Set placeholder as src
			$attr['src'] = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';

			// Move srcset to data-srcset if present
			if ( isset( $attr['srcset'] ) ) {
				$attr['data-srcset'] = $attr['srcset'];
				unset( $attr['srcset'] );
			}

			// Add lazy loading class
			if ( isset( $attr['class'] ) ) {
				$attr['class'] .= ' sgo-lazy';
			} else {
				$attr['class'] = 'sgo-lazy';
			}
		}

		return $attr;
	}

	/**
	 * Add srcset attributes for responsive images
	 *
	 * @param array  $attr Image attributes
	 * @param object $attachment Attachment object
	 * @param string $size Image size
	 * @return array Modified attributes
	 */
	public function add_srcset_attributes( $attr, $attachment, $size ) {
		// Ensure srcset is enabled
		if ( ! isset( $attr['srcset'] ) ) {
			return $attr;
		}

		return $attr;
	}

	/**
	 * Check if browser supports WebP
	 *
	 * @return bool Browser support status
	 */
	private function browser_supports_webp() {
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$http_accept = isset( $_SERVER['HTTP_ACCEPT'] ) ? wp_unslash( $_SERVER['HTTP_ACCEPT'] ) : '';
		return strpos( $http_accept, 'image/webp' ) !== false;
	}

	/**
	 * Check if browser supports AVIF
	 *
	 * @return bool Browser support status
	 */
	private function browser_supports_avif() {
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
		$http_accept = isset( $_SERVER['HTTP_ACCEPT'] ) ? wp_unslash( $_SERVER['HTTP_ACCEPT'] ) : '';
		return strpos( $http_accept, 'image/avif' ) !== false;
	}

	/**
	 * Get converted image paths for an attachment
	 *
	 * @param int $attachment_id Attachment ID
	 * @return array Array of converted image paths
	 */
	public function get_converted_images( $attachment_id ) {
		$original_path = get_attached_file( $attachment_id );
		$file_info = pathinfo( $original_path );

		$converted = array(
			'original' => $original_path,
		);

		// Check for WebP
		$webp_path = $file_info['dirname'] . '/' . $file_info['filename'] . '.webp';
		if ( file_exists( $webp_path ) ) {
			$converted['webp'] = $webp_path;
		}

		// Check for AVIF
		$avif_path = $file_info['dirname'] . '/' . $file_info['filename'] . '.avif';
		if ( file_exists( $avif_path ) ) {
			$converted['avif'] = $avif_path;
		}

		return $converted;
	}
}
