<?php
/**
 * Gallery Manager Class
 *
 * Handles gallery custom post type registration and management
 *
 * @package Smart_Gallery_Optimizer
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class SGO_Gallery_Manager
 *
 * Manages gallery custom post type and admin interface
 */
class SGO_Gallery_Manager {

	/**
	 * Custom post type name
	 *
	 * @var string
	 */
	const POST_TYPE = 'sgo_gallery';

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'register_post_type' ) );
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );

		// AJAX handlers
		add_action( 'wp_ajax_sgo_save_gallery', array( $this, 'ajax_save_gallery' ) );
		add_action( 'wp_ajax_sgo_delete_gallery', array( $this, 'ajax_delete_gallery' ) );
		add_action( 'wp_ajax_sgo_duplicate_gallery', array( $this, 'ajax_duplicate_gallery' ) );
	}

	/**
	 * Register custom post type for galleries
	 */
	public function register_post_type() {
		$labels = array(
			'name'               => _x( 'Galleries', 'post type general name', 'smart-gallery-optimizer' ),
			'singular_name'      => _x( 'Gallery', 'post type singular name', 'smart-gallery-optimizer' ),
			'menu_name'          => _x( 'Galleries', 'admin menu', 'smart-gallery-optimizer' ),
			'name_admin_bar'     => _x( 'Gallery', 'add new on admin bar', 'smart-gallery-optimizer' ),
			'add_new'            => _x( 'Add New', 'gallery', 'smart-gallery-optimizer' ),
			'add_new_item'       => __( 'Add New Gallery', 'smart-gallery-optimizer' ),
			'new_item'           => __( 'New Gallery', 'smart-gallery-optimizer' ),
			'edit_item'          => __( 'Edit Gallery', 'smart-gallery-optimizer' ),
			'view_item'          => __( 'View Gallery', 'smart-gallery-optimizer' ),
			'all_items'          => __( 'All Galleries', 'smart-gallery-optimizer' ),
			'search_items'       => __( 'Search Galleries', 'smart-gallery-optimizer' ),
			'parent_item_colon'  => __( 'Parent Galleries:', 'smart-gallery-optimizer' ),
			'not_found'          => __( 'No galleries found.', 'smart-gallery-optimizer' ),
			'not_found_in_trash' => __( 'No galleries found in Trash.', 'smart-gallery-optimizer' ),
		);

		$args = array(
			'labels'              => $labels,
			'description'         => __( 'Smart Gallery Collections', 'smart-gallery-optimizer' ),
			'public'              => false,
			'publicly_queryable'  => false,
			'show_ui'             => false,
			'show_in_menu'        => false,
			'query_var'           => false,
			'rewrite'             => false,
			'capability_type'     => 'post',
			'has_archive'         => false,
			'hierarchical'        => false,
			'menu_position'       => null,
			'supports'            => array( 'title', 'custom-fields' ),
		);

		register_post_type( self::POST_TYPE, $args );
	}

	/**
	 * Add admin menu pages
	 */
	public function add_admin_menu() {
		// Add Gallery List submenu
		add_submenu_page(
			'smart-gallery-optimizer',
			__( 'Gallery List', 'smart-gallery-optimizer' ),
			__( 'Gallery List', 'smart-gallery-optimizer' ),
			'manage_options',
			'sgo-gallery-list',
			array( $this, 'render_gallery_list_page' )
		);

		// Add Gallery Editor page (hidden from menu)
		add_submenu_page(
			null, // No parent = hidden from menu
			__( 'Edit Gallery', 'smart-gallery-optimizer' ),
			__( 'Edit Gallery', 'smart-gallery-optimizer' ),
			'manage_options',
			'sgo-gallery-editor',
			array( $this, 'render_gallery_editor_page' )
		);
	}

	/**
	 * Enqueue admin assets
	 *
	 * @param string $hook Current admin page hook
	 */
	public function enqueue_admin_assets( $hook ) {
		// Only load on our admin pages
		if ( strpos( $hook, 'sgo-gallery' ) === false ) {
			return;
		}

		// Enqueue WordPress media library
		wp_enqueue_media();

		// Enqueue jQuery UI for drag and drop
		wp_enqueue_script( 'jquery-ui-sortable' );

		// Enqueue our custom CSS
		wp_enqueue_style(
			'sgo-gallery-manager',
			SGO_PLUGIN_URL . 'assets/css/gallery-manager.css',
			array(),
			SGO_VERSION
		);

		// Enqueue our custom JS
		wp_enqueue_script(
			'sgo-gallery-manager',
			SGO_PLUGIN_URL . 'assets/js/gallery-manager.js',
			array( 'jquery', 'jquery-ui-sortable' ),
			SGO_VERSION,
			true
		);

		// Localize script with AJAX URL and nonce
		wp_localize_script(
			'sgo-gallery-manager',
			'sgoGallery',
			array(
				'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
				'nonce'         => wp_create_nonce( 'sgo_gallery_nonce' ),
				'confirmDelete' => __( 'Are you sure you want to delete this gallery?', 'smart-gallery-optimizer' ),
				'copySuccess'   => __( 'Shortcode copied to clipboard!', 'smart-gallery-optimizer' ),
				'copyError'     => __( 'Failed to copy shortcode. Please copy manually.', 'smart-gallery-optimizer' ),
				'saveSuccess'   => __( 'Gallery saved successfully!', 'smart-gallery-optimizer' ),
				'saveError'     => __( 'Failed to save gallery. Please try again.', 'smart-gallery-optimizer' ),
				'deleteSuccess' => __( 'Gallery deleted successfully!', 'smart-gallery-optimizer' ),
				'deleteError'   => __( 'Failed to delete gallery. Please try again.', 'smart-gallery-optimizer' ),
			)
		);
	}

	/**
	 * Render gallery list page
	 */
	public function render_gallery_list_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'smart-gallery-optimizer' ) );
		}

		require_once SGO_PLUGIN_DIR . 'admin/pages/gallery-list.php';
	}

	/**
	 * Render gallery editor page
	 */
	public function render_gallery_editor_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'smart-gallery-optimizer' ) );
		}

		require_once SGO_PLUGIN_DIR . 'admin/pages/gallery-editor.php';
	}

	/**
	 * AJAX handler to save gallery
	 */
	public function ajax_save_gallery() {
		// Verify nonce
		check_ajax_referer( 'sgo_gallery_nonce', 'nonce' );

		// Check capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'smart-gallery-optimizer' ) ) );
		}

		// Get and sanitize POST data
		$gallery_id   = isset( $_POST['gallery_id'] ) ? absint( $_POST['gallery_id'] ) : 0;
		$gallery_name = isset( $_POST['gallery_name'] ) ? sanitize_text_field( wp_unslash( $_POST['gallery_name'] ) ) : '';
		$image_ids    = ( isset( $_POST['image_ids'] ) && is_array( $_POST['image_ids'] ) ) ? array_map( 'absint', wp_unslash( $_POST['image_ids'] ) ) : array();
		$columns      = isset( $_POST['columns'] ) ? absint( $_POST['columns'] ) : 3;
		$image_size   = isset( $_POST['image_size'] ) ? sanitize_key( wp_unslash( $_POST['image_size'] ) ) : 'medium';
		$gap          = isset( $_POST['gap'] ) ? absint( $_POST['gap'] ) : 20;
		$link_type    = isset( $_POST['link_type'] ) ? sanitize_key( wp_unslash( $_POST['link_type'] ) ) : 'none';
		$lb_title     = isset( $_POST['lightbox_title'] ) ? (bool) wp_unslash( $_POST['lightbox_title'] ) : false;
		$lb_caption   = isset( $_POST['lightbox_caption'] ) ? (bool) wp_unslash( $_POST['lightbox_caption'] ) : false;

		// Validate required fields
		if ( empty( $gallery_name ) ) {
			wp_send_json_error( array( 'message' => __( 'Gallery name is required', 'smart-gallery-optimizer' ) ) );
		}

		// Validate columns range
		if ( $columns < 2 || $columns > 6 ) {
			$columns = 3;
		}

		// Validate image size
		$allowed_sizes = array( 'thumbnail', 'medium', 'large', 'full' );
		if ( ! in_array( $image_size, $allowed_sizes, true ) ) {
			$image_size = 'medium';
		}

		// Validate link type
		$allowed_link_types = array( 'none', 'direct', 'lightbox' );
		if ( ! in_array( $link_type, $allowed_link_types, true ) ) {
			$link_type = 'none';
		}

		// Create or update post
		$post_data = array(
			'post_title'  => $gallery_name,
			'post_type'   => self::POST_TYPE,
			'post_status' => 'publish',
		);

		if ( $gallery_id > 0 ) {
			$post_data['ID'] = $gallery_id;
			$post_id = wp_update_post( $post_data );
		} else {
			$post_id = wp_insert_post( $post_data );
		}

		if ( is_wp_error( $post_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Failed to save gallery', 'smart-gallery-optimizer' ) ) );
		}

		// Save post meta
		update_post_meta( $post_id, '_sgo_gallery_images', $image_ids );
		update_post_meta( $post_id, '_sgo_gallery_columns', $columns );
		update_post_meta( $post_id, '_sgo_gallery_size', $image_size );
		update_post_meta( $post_id, '_sgo_gallery_gap', $gap );
		update_post_meta( $post_id, '_sgo_gallery_link_type', $link_type );
		update_post_meta( $post_id, '_sgo_gallery_lightbox_title', $lb_title );
		update_post_meta( $post_id, '_sgo_gallery_lightbox_caption', $lb_caption );

		// Return success with gallery ID
		wp_send_json_success(
			array(
				'message'    => __( 'Gallery saved successfully!', 'smart-gallery-optimizer' ),
				'gallery_id' => $post_id,
				'shortcode'  => '[smart_gallery id="' . $post_id . '"]',
			)
		);
	}

	/**
	 * AJAX handler to delete gallery
	 */
	public function ajax_delete_gallery() {
		// Verify nonce
		check_ajax_referer( 'sgo_gallery_nonce', 'nonce' );

		// Check capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'smart-gallery-optimizer' ) ) );
		}

		// Get gallery ID
		$gallery_id = isset( $_POST['gallery_id'] ) ? absint( $_POST['gallery_id'] ) : 0;

		if ( ! $gallery_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid gallery ID', 'smart-gallery-optimizer' ) ) );
		}

		// Delete the post
		$result = wp_delete_post( $gallery_id, true );

		if ( ! $result ) {
			wp_send_json_error( array( 'message' => __( 'Failed to delete gallery', 'smart-gallery-optimizer' ) ) );
		}

		wp_send_json_success( array( 'message' => __( 'Gallery deleted successfully!', 'smart-gallery-optimizer' ) ) );
	}

	/**
	 * AJAX handler to duplicate gallery
	 */
	public function ajax_duplicate_gallery() {
		// Verify nonce
		check_ajax_referer( 'sgo_gallery_nonce', 'nonce' );

		// Check capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'smart-gallery-optimizer' ) ) );
		}

		// Get gallery ID
		$gallery_id = isset( $_POST['gallery_id'] ) ? absint( $_POST['gallery_id'] ) : 0;

		if ( ! $gallery_id ) {
			wp_send_json_error( array( 'message' => __( 'Invalid gallery ID', 'smart-gallery-optimizer' ) ) );
		}

		// Get original post
		$original_post = get_post( $gallery_id );

		if ( ! $original_post || $original_post->post_type !== self::POST_TYPE ) {
			wp_send_json_error( array( 'message' => __( 'Gallery not found', 'smart-gallery-optimizer' ) ) );
		}

		// Create duplicate
		$new_post_data = array(
			'post_title'  => $original_post->post_title . ' (Copy)',
			'post_type'   => self::POST_TYPE,
			'post_status' => 'publish',
		);

		$new_post_id = wp_insert_post( $new_post_data );

		if ( is_wp_error( $new_post_id ) ) {
			wp_send_json_error( array( 'message' => __( 'Failed to duplicate gallery', 'smart-gallery-optimizer' ) ) );
		}

		// Copy all meta data
		$meta_keys = array(
			'_sgo_gallery_images',
			'_sgo_gallery_columns',
			'_sgo_gallery_size',
			'_sgo_gallery_gap',
			'_sgo_gallery_link_type',
			'_sgo_gallery_lightbox_title',
			'_sgo_gallery_lightbox_caption',
		);

		foreach ( $meta_keys as $meta_key ) {
			$meta_value = get_post_meta( $gallery_id, $meta_key, true );
			update_post_meta( $new_post_id, $meta_key, $meta_value );
		}

		wp_send_json_success(
			array(
				'message'    => __( 'Gallery duplicated successfully!', 'smart-gallery-optimizer' ),
				'gallery_id' => $new_post_id,
			)
		);
	}

	/**
	 * Get gallery data by ID
	 *
	 * @param int $gallery_id Gallery post ID
	 * @return array|false Gallery data or false if not found
	 */
	public static function get_gallery_data( $gallery_id ) {
		$post = get_post( $gallery_id );

		if ( ! $post || $post->post_type !== self::POST_TYPE ) {
			return false;
		}

		// Get meta values and convert to proper boolean for checkbox values
		$lightbox_title_meta = get_post_meta( $post->ID, '_sgo_gallery_lightbox_title', true );
		$lightbox_caption_meta = get_post_meta( $post->ID, '_sgo_gallery_lightbox_caption', true );

		return array(
			'id'              => $post->ID,
			'name'            => $post->post_title,
			'images'          => get_post_meta( $post->ID, '_sgo_gallery_images', true ),
			'columns'         => get_post_meta( $post->ID, '_sgo_gallery_columns', true ),
			'size'            => get_post_meta( $post->ID, '_sgo_gallery_size', true ),
			'gap'             => get_post_meta( $post->ID, '_sgo_gallery_gap', true ),
			'link_type'       => get_post_meta( $post->ID, '_sgo_gallery_link_type', true ),
			'lightbox_title'  => ! empty( $lightbox_title_meta ) && $lightbox_title_meta !== '0',
			'lightbox_caption' => ! empty( $lightbox_caption_meta ) && $lightbox_caption_meta !== '0',
		);
	}

	/**
	 * Get all galleries
	 *
	 * @param array $args Query arguments
	 * @return array Array of gallery data
	 */
	public static function get_all_galleries( $args = array() ) {
		$defaults = array(
			'post_type'      => self::POST_TYPE,
			'post_status'    => 'publish',
			'posts_per_page' => 20,
			'paged'          => 1,
			'orderby'        => 'date',
			'order'          => 'DESC',
		);

		$args = wp_parse_args( $args, $defaults );

		$query = new WP_Query( $args );
		$galleries = array();

		if ( $query->have_posts() ) {
			while ( $query->have_posts() ) {
				$query->the_post();
				$post_id = get_the_ID();
				$images = get_post_meta( $post_id, '_sgo_gallery_images', true );

				$galleries[] = array(
					'id'          => $post_id,
					'name'        => get_the_title(),
					'image_count' => is_array( $images ) ? count( $images ) : 0,
					'columns'     => get_post_meta( $post_id, '_sgo_gallery_columns', true ),
					'size'        => get_post_meta( $post_id, '_sgo_gallery_size', true ),
					'link_type'   => get_post_meta( $post_id, '_sgo_gallery_link_type', true ),
					'shortcode'   => '[smart_gallery id="' . $post_id . '"]',
				);
			}
			wp_reset_postdata();
		}

		return array(
			'galleries'   => $galleries,
			'total'       => $query->found_posts,
			'total_pages' => $query->max_num_pages,
		);
	}
}
