<?php
/**
 * EXIF Handler Class
 *
 * Handles EXIF data extraction and automatic alt-text generation
 *
 * @package Smart_Gallery_Optimizer
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class SGO_EXIF_Handler
 *
 * Manages EXIF data and automatic alt-text generation
 */
class SGO_EXIF_Handler {

	/**
	 * Constructor
	 */
	public function __construct() {
		// Hook into attachment metadata generation
		add_filter( 'wp_generate_attachment_metadata', array( $this, 'extract_and_save_exif' ), 10, 2 );

		// Hook to generate alt-text after upload
		add_action( 'add_attachment', array( $this, 'auto_generate_alt_text' ) );

		// Admin menu for bulk alt-text generation
		add_action( 'admin_menu', array( $this, 'add_bulk_alttext_page' ) );

		// AJAX handler for bulk generation
		add_action( 'wp_ajax_sgo_bulk_generate_alttext', array( $this, 'ajax_bulk_generate_alttext' ) );
	}

	/**
	 * Extract EXIF data and save to post meta
	 *
	 * @param array $metadata      Attachment metadata
	 * @param int   $attachment_id Attachment post ID
	 * @return array Modified metadata
	 */
	public function extract_and_save_exif( $metadata, $attachment_id ) {
		// Check if EXIF functions are available
		if ( ! function_exists( 'exif_read_data' ) ) {
			return $metadata;
		}

		// Get file path
		$file_path = get_attached_file( $attachment_id );

		if ( ! $file_path || ! file_exists( $file_path ) ) {
			return $metadata;
		}

		// Only process image files
		$mime_type = get_post_mime_type( $attachment_id );
		if ( strpos( $mime_type, 'image/' ) !== 0 ) {
			return $metadata;
		}

		// Read EXIF data
		$exif = @exif_read_data( $file_path, 0, true );

		if ( ! $exif ) {
			return $metadata;
		}

		// Extract relevant EXIF data
		$exif_data = array();

		// Camera information
		if ( isset( $exif['IFD0']['Make'] ) ) {
			$exif_data['camera_make'] = trim( $exif['IFD0']['Make'] );
		}

		if ( isset( $exif['IFD0']['Model'] ) ) {
			$exif_data['camera_model'] = trim( $exif['IFD0']['Model'] );
		}

		// Date taken
		if ( isset( $exif['EXIF']['DateTimeOriginal'] ) ) {
			$exif_data['date_taken'] = $exif['EXIF']['DateTimeOriginal'];
		} elseif ( isset( $exif['IFD0']['DateTime'] ) ) {
			$exif_data['date_taken'] = $exif['IFD0']['DateTime'];
		}

		// GPS location
		if ( isset( $exif['GPS']['GPSLatitude'] ) && isset( $exif['GPS']['GPSLongitude'] ) ) {
			$lat = $this->convert_gps_to_decimal( $exif['GPS']['GPSLatitude'], $exif['GPS']['GPSLatitudeRef'] );
			$lon = $this->convert_gps_to_decimal( $exif['GPS']['GPSLongitude'], $exif['GPS']['GPSLongitudeRef'] );

			if ( $lat && $lon ) {
				$exif_data['gps_latitude'] = $lat;
				$exif_data['gps_longitude'] = $lon;
				$exif_data['gps_location'] = $this->get_location_from_coords( $lat, $lon );
			}
		}

		// Image description
		if ( isset( $exif['IFD0']['ImageDescription'] ) ) {
			$exif_data['description'] = trim( $exif['IFD0']['ImageDescription'] );
		}

		// Orientation
		if ( isset( $exif['IFD0']['Orientation'] ) ) {
			$exif_data['orientation'] = $exif['IFD0']['Orientation'];
		}

		// Save EXIF data to post meta
		if ( ! empty( $exif_data ) ) {
			update_post_meta( $attachment_id, '_sgo_exif_data', $exif_data );
		}

		return $metadata;
	}

	/**
	 * Automatically generate alt-text for new attachments
	 *
	 * @param int $attachment_id Attachment post ID
	 */
	public function auto_generate_alt_text( $attachment_id ) {
		// Check if auto alt-text is enabled
		$settings = get_option( 'sgo_settings', array() );
		$auto_alttext = isset( $settings['auto_alttext_enabled'] ) && $settings['auto_alttext_enabled'];

		if ( ! $auto_alttext ) {
			return;
		}

		// Check if alt-text already exists
		$existing_alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );

		if ( ! empty( $existing_alt ) ) {
			return;
		}

		// Generate and save alt-text
		$alt_text = $this->generate_alt_text( $attachment_id );

		if ( $alt_text ) {
			update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt_text );
		}
	}

	/**
	 * Generate alt-text for an image
	 *
	 * @param int $attachment_id Attachment post ID
	 * @return string Generated alt-text
	 */
	public function generate_alt_text( $attachment_id ) {
		// Priority 1: Use existing alt-text if present
		$existing_alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		if ( ! empty( $existing_alt ) ) {
			return $existing_alt;
		}

		// Get settings
		$settings = get_option( 'sgo_settings', array() );
		$template = isset( $settings['alttext_template'] ) ? $settings['alttext_template'] : '{description} - Photo taken with {camera} on {date}';
		$include_camera = isset( $settings['alttext_include_camera'] ) && $settings['alttext_include_camera'];
		$include_date = isset( $settings['alttext_include_date'] ) && $settings['alttext_include_date'];
		$include_location = isset( $settings['alttext_include_location'] ) && $settings['alttext_include_location'];

		// Priority 2: Generate from EXIF data
		$exif_data = get_post_meta( $attachment_id, '_sgo_exif_data', true );

		if ( is_array( $exif_data ) && ! empty( $exif_data ) ) {
			$alt_text = $template;

			// Replace description
			if ( isset( $exif_data['description'] ) && ! empty( $exif_data['description'] ) ) {
				$description = sanitize_text_field( $exif_data['description'] );
			} else {
				$description = get_the_title( $attachment_id );
			}
			$alt_text = str_replace( '{description}', $description, $alt_text );

			// Replace camera info
			if ( $include_camera && isset( $exif_data['camera_make'] ) && isset( $exif_data['camera_model'] ) ) {
				$camera = $exif_data['camera_make'] . ' ' . $exif_data['camera_model'];
				$alt_text = str_replace( '{camera}', $camera, $alt_text );
			} else {
				$alt_text = str_replace( ' - Photo taken with {camera}', '', $alt_text );
				$alt_text = str_replace( '{camera}', '', $alt_text );
			}

			// Replace date
			if ( $include_date && isset( $exif_data['date_taken'] ) ) {
				$date = date_i18n( get_option( 'date_format' ), strtotime( $exif_data['date_taken'] ) );
				$alt_text = str_replace( '{date}', $date, $alt_text );
			} else {
				$alt_text = str_replace( ' on {date}', '', $alt_text );
				$alt_text = str_replace( '{date}', '', $alt_text );
			}

			// Replace location
			if ( $include_location && isset( $exif_data['gps_location'] ) ) {
				$alt_text = str_replace( '{location}', $exif_data['gps_location'], $alt_text );
			} else {
				$alt_text = str_replace( ' at {location}', '', $alt_text );
				$alt_text = str_replace( '{location}', '', $alt_text );
			}

			// Clean up extra spaces and dashes
			$alt_text = preg_replace( '/\s+/', ' ', $alt_text );
			$alt_text = trim( $alt_text, ' -' );

			if ( ! empty( $alt_text ) ) {
				return $alt_text;
			}
		}

		// Priority 3: Use image title
		$title = get_the_title( $attachment_id );
		if ( ! empty( $title ) ) {
			return $this->clean_filename( $title );
		}

		// Priority 4: Use filename (cleaned up)
		$filename = basename( get_attached_file( $attachment_id ) );
		$filename = preg_replace( '/\.[^.]+$/', '', $filename ); // Remove extension
		return $this->clean_filename( $filename );
	}

	/**
	 * Clean filename for use as alt-text
	 *
	 * @param string $filename Filename to clean
	 * @return string Cleaned filename
	 */
	private function clean_filename( $filename ) {
		// Replace dashes and underscores with spaces
		$clean = str_replace( array( '-', '_' ), ' ', $filename );

		// Remove file extension
		$clean = preg_replace( '/\.[^.]+$/', '', $clean );

		// Remove numbers at the end (like IMG-1234)
		$clean = preg_replace( '/\s*\d+\s*$/', '', $clean );

		// Capitalize first letter of each word
		$clean = ucwords( strtolower( $clean ) );

		// Clean up extra spaces
		$clean = preg_replace( '/\s+/', ' ', $clean );

		return trim( $clean );
	}

	/**
	 * Convert GPS coordinates to decimal format
	 *
	 * @param array  $coord GPS coordinate array
	 * @param string $ref   GPS reference (N/S/E/W)
	 * @return float|false Decimal coordinate or false
	 */
	private function convert_gps_to_decimal( $coord, $ref ) {
		if ( ! is_array( $coord ) || count( $coord ) < 3 ) {
			return false;
		}

		$degrees = $this->gps_fraction_to_decimal( $coord[0] );
		$minutes = $this->gps_fraction_to_decimal( $coord[1] );
		$seconds = $this->gps_fraction_to_decimal( $coord[2] );

		$decimal = $degrees + ( $minutes / 60 ) + ( $seconds / 3600 );

		if ( $ref === 'S' || $ref === 'W' ) {
			$decimal = -$decimal;
		}

		return $decimal;
	}

	/**
	 * Convert GPS fraction to decimal
	 *
	 * @param string $fraction GPS fraction (e.g., "123/1")
	 * @return float Decimal value
	 */
	private function gps_fraction_to_decimal( $fraction ) {
		if ( strpos( $fraction, '/' ) !== false ) {
			$parts = explode( '/', $fraction );
			if ( count( $parts ) === 2 && $parts[1] != 0 ) {
				return $parts[0] / $parts[1];
			}
		}
		return (float) $fraction;
	}

	/**
	 * Get location name from GPS coordinates
	 *
	 * @param float $lat Latitude
	 * @param float $lon Longitude
	 * @return string Location name or coordinates
	 */
	private function get_location_from_coords( $lat, $lon ) {
		// For basic implementation, just return formatted coordinates
		// In production, you might want to use a geocoding service
		return sprintf( '%.6f, %.6f', $lat, $lon );
	}

	/**
	 * Add bulk alt-text generation page to admin menu
	 */
	public function add_bulk_alttext_page() {
		add_submenu_page(
			'smart-gallery-optimizer',
			__( 'Bulk Alt-Text Generator', 'smart-gallery-optimizer' ),
			__( 'Bulk Alt-Text', 'smart-gallery-optimizer' ),
			'manage_options',
			'sgo-bulk-alttext',
			array( $this, 'render_bulk_alttext_page' )
		);
	}

	/**
	 * Render bulk alt-text generation page
	 */
	public function render_bulk_alttext_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'smart-gallery-optimizer' ) );
		}

		// Get images without alt-text
		$images = get_posts(
			array(
				'post_type'      => 'attachment',
				'post_mime_type' => 'image',
				'posts_per_page' => 100,
				'meta_query'     => array(
					'relation' => 'OR',
					array(
						'key'     => '_wp_attachment_image_alt',
						'compare' => 'NOT EXISTS',
					),
					array(
						'key'     => '_wp_attachment_image_alt',
						'value'   => '',
						'compare' => '=',
					),
				),
			)
		);

		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Bulk Alt-Text Generator', 'smart-gallery-optimizer' ); ?></h1>
			<p class="description">
				<?php esc_html_e( 'Generate alt-text for images that are missing it. The alt-text will be generated using EXIF data, image titles, or filenames.', 'smart-gallery-optimizer' ); ?>
			</p>

			<?php if ( ! empty( $images ) ) : ?>
				<p>
					<strong>
						<?php
						printf(
							/* translators: %d: number of images */
							esc_html( _n( 'Found %d image without alt-text.', 'Found %d images without alt-text.', count( $images ), 'smart-gallery-optimizer' ) ),
							count( $images )
						);
						?>
					</strong>
				</p>

				<table class="wp-list-table widefat fixed striped" id="sgo-alttext-preview">
					<thead>
						<tr>
							<th style="width: 60px;"><?php esc_html_e( 'Image', 'smart-gallery-optimizer' ); ?></th>
							<th><?php esc_html_e( 'Filename', 'smart-gallery-optimizer' ); ?></th>
							<th><?php esc_html_e( 'Generated Alt-Text', 'smart-gallery-optimizer' ); ?></th>
						</tr>
					</thead>
					<tbody>
						<?php foreach ( $images as $image ) : ?>
							<?php
							$alt_text = $this->generate_alt_text( $image->ID );
							$thumbnail = wp_get_attachment_image_url( $image->ID, 'thumbnail' );
							?>
							<tr data-id="<?php echo esc_attr( $image->ID ); ?>">
								<td>
									<?php if ( $thumbnail ) : ?>
										<img src="<?php echo esc_url( $thumbnail ); ?>" style="width: 50px; height: 50px; object-fit: cover;">
									<?php endif; ?>
								</td>
								<td><?php echo esc_html( basename( get_attached_file( $image->ID ) ) ); ?></td>
								<td class="preview-alttext"><?php echo esc_html( $alt_text ); ?></td>
							</tr>
						<?php endforeach; ?>
					</tbody>
				</table>

				<p style="margin-top: 20px;">
					<button type="button" id="sgo-bulk-generate" class="button button-primary button-large">
						<?php esc_html_e( 'Generate Alt-Text for All Images', 'smart-gallery-optimizer' ); ?>
					</button>
					<span id="sgo-bulk-status" style="margin-left: 15px;"></span>
				</p>

				<script>
				jQuery(document).ready(function($) {
					$('#sgo-bulk-generate').on('click', function() {
						var $button = $(this);
						var $status = $('#sgo-bulk-status');
						var imageIds = [];

						$('#sgo-alttext-preview tbody tr').each(function() {
							imageIds.push($(this).data('id'));
						});

						$button.prop('disabled', true).text('Generating...');
						$status.html('<span style="color: #2271b1;">Processing...</span>');

						$.ajax({
							url: ajaxurl,
							type: 'POST',
							data: {
								action: 'sgo_bulk_generate_alttext',
								nonce: '<?php echo esc_js( wp_create_nonce( 'sgo_bulk_alttext' ) ); ?>',
								image_ids: imageIds
							},
							success: function(response) {
								if (response.success) {
									$status.html('<span style="color: #00703c;">✓ ' + response.data.message + '</span>');
									setTimeout(function() {
										location.reload();
									}, 2000);
								} else {
									$status.html('<span style="color: #b32d2e;">✗ ' + response.data.message + '</span>');
									$button.prop('disabled', false).text('Generate Alt-Text for All Images');
								}
							},
							error: function() {
								$status.html('<span style="color: #b32d2e;">✗ Error generating alt-text</span>');
								$button.prop('disabled', false).text('Generate Alt-Text for All Images');
							}
						});
					});
				});
				</script>

			<?php else : ?>
				<div class="notice notice-success">
					<p><?php esc_html_e( 'Great! All your images have alt-text.', 'smart-gallery-optimizer' ); ?></p>
				</div>
			<?php endif; ?>
		</div>
		<?php
	}

	/**
	 * AJAX handler for bulk alt-text generation
	 */
	public function ajax_bulk_generate_alttext() {
		check_ajax_referer( 'sgo_bulk_alttext', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => __( 'Insufficient permissions', 'smart-gallery-optimizer' ) ) );
		}

		$image_ids = ( isset( $_POST['image_ids'] ) && is_array( $_POST['image_ids'] ) ) ? array_map( 'absint', wp_unslash( $_POST['image_ids'] ) ) : array();

		if ( empty( $image_ids ) ) {
			wp_send_json_error( array( 'message' => __( 'No images to process', 'smart-gallery-optimizer' ) ) );
		}

		$count = 0;
		foreach ( $image_ids as $image_id ) {
			$alt_text = $this->generate_alt_text( $image_id );
			if ( $alt_text ) {
				update_post_meta( $image_id, '_wp_attachment_image_alt', $alt_text );
				$count++;
			}
		}

		wp_send_json_success(
			array(
				'message' => sprintf(
					/* translators: %d: number of images */
					_n( 'Generated alt-text for %d image', 'Generated alt-text for %d images', $count, 'smart-gallery-optimizer' ),
					$count
				),
			)
		);
	}
}
