<?php
/**
 * Admin Settings Class
 *
 * Handles plugin settings and admin dashboard
 *
 * @package Smart_Gallery_Optimizer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SGO_Admin_Settings Class
 */
class SGO_Admin_Settings {

	/**
	 * Settings page slug
	 *
	 * @var string
	 */
	private $page_slug = 'smart-gallery-optimizer';

	/**
	 * Option name for settings
	 *
	 * @var string
	 */
	private $option_name = 'sgo_settings';

	/**
	 * Constructor
	 */
	public function __construct() {
		// Add admin menu
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );

		// Register settings
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Enqueue admin scripts and styles
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
	}

	/**
	 * Add admin menu item
	 */
	public function add_admin_menu() {
		add_menu_page(
			__( 'Smart Gallery Optimizer', 'smart-gallery-optimizer' ),
			__( 'Smart Gallery', 'smart-gallery-optimizer' ),
			'manage_options',
			$this->page_slug,
			array( $this, 'render_settings_page' ),
			'dashicons-images-alt2',
			30
		);
	}

	/**
	 * Register plugin settings
	 */
	public function register_settings() {
		// Register settings
		register_setting(
			$this->option_name . '_group',
			$this->option_name,
			array( $this, 'sanitize_settings' )
		);

		// Add settings section
		add_settings_section(
			'sgo_conversion_section',
			__( 'Image Conversion Settings', 'smart-gallery-optimizer' ),
			array( $this, 'render_conversion_section' ),
			$this->page_slug
		);

		// Add WebP setting field
		add_settings_field(
			'webp_conversion',
			__( 'WebP Conversion', 'smart-gallery-optimizer' ),
			array( $this, 'render_webp_field' ),
			$this->page_slug,
			'sgo_conversion_section'
		);

		// Add AVIF setting field
		add_settings_field(
			'avif_conversion',
			__( 'AVIF Conversion', 'smart-gallery-optimizer' ),
			array( $this, 'render_avif_field' ),
			$this->page_slug,
			'sgo_conversion_section'
		);

		// Add WebP quality field
		add_settings_field(
			'webp_quality',
			__( 'WebP Quality', 'smart-gallery-optimizer' ),
			array( $this, 'render_webp_quality_field' ),
			$this->page_slug,
			'sgo_conversion_section'
		);

		// Add AVIF quality field
		add_settings_field(
			'avif_quality',
			__( 'AVIF Quality', 'smart-gallery-optimizer' ),
			array( $this, 'render_avif_quality_field' ),
			$this->page_slug,
			'sgo_conversion_section'
		);

		// Add Performance & Display settings section
		add_settings_section(
			'sgo_performance_section',
			__( 'Performance &amp; Display Settings', 'smart-gallery-optimizer' ),
			array( $this, 'render_performance_section' ),
			$this->page_slug
		);

		// Add Lazy Loading field
		add_settings_field(
			'lazy_loading',
			__( 'Enable Lazy Loading', 'smart-gallery-optimizer' ),
			array( $this, 'render_lazy_loading_field' ),
			$this->page_slug,
			'sgo_performance_section'
		);

		// Add Smart Grid field
		add_settings_field(
			'smart_grid_enabled',
			__( 'Enable Smart Grid', 'smart-gallery-optimizer' ),
			array( $this, 'render_smart_grid_field' ),
			$this->page_slug,
			'sgo_performance_section'
		);

		// Add Default Columns field
		add_settings_field(
			'default_columns',
			__( 'Default Columns', 'smart-gallery-optimizer' ),
			array( $this, 'render_default_columns_field' ),
			$this->page_slug,
			'sgo_performance_section'
		);

		// Add EXIF & SEO settings section
		add_settings_section(
			'sgo_exif_section',
			__( 'EXIF &amp; SEO Settings', 'smart-gallery-optimizer' ),
			array( $this, 'render_exif_section' ),
			$this->page_slug
		);

		// Auto alt-text field
		add_settings_field(
			'auto_alttext_enabled',
			__( 'Enable Auto Alt-Text', 'smart-gallery-optimizer' ),
			array( $this, 'render_auto_alttext_field' ),
			$this->page_slug,
			'sgo_exif_section'
		);

		// Include camera info field
		add_settings_field(
			'alttext_include_camera',
			__( 'Include Camera Info', 'smart-gallery-optimizer' ),
			array( $this, 'render_alttext_camera_field' ),
			$this->page_slug,
			'sgo_exif_section'
		);

		// Include date info field
		add_settings_field(
			'alttext_include_date',
			__( 'Include Date Info', 'smart-gallery-optimizer' ),
			array( $this, 'render_alttext_date_field' ),
			$this->page_slug,
			'sgo_exif_section'
		);

		// Include location info field
		add_settings_field(
			'alttext_include_location',
			__( 'Include Location Info', 'smart-gallery-optimizer' ),
			array( $this, 'render_alttext_location_field' ),
			$this->page_slug,
			'sgo_exif_section'
		);

		// Alt-text template field
		add_settings_field(
			'alttext_template',
			__( 'Alt-Text Template', 'smart-gallery-optimizer' ),
			array( $this, 'render_alttext_template_field' ),
			$this->page_slug,
			'sgo_exif_section'
		);

		// Add Link Behavior settings section
		add_settings_section(
			'sgo_link_section',
			__( 'Link Behavior Settings', 'smart-gallery-optimizer' ),
			array( $this, 'render_link_section' ),
			$this->page_slug
		);

		// Default link type field
		add_settings_field(
			'default_link_type',
			__( 'Default Link Type', 'smart-gallery-optimizer' ),
			array( $this, 'render_link_type_field' ),
			$this->page_slug,
			'sgo_link_section'
		);

		// Open in new tab field
		add_settings_field(
			'open_links_new_tab',
			__( 'Open Direct Links in New Tab', 'smart-gallery-optimizer' ),
			array( $this, 'render_new_tab_field' ),
			$this->page_slug,
			'sgo_link_section'
		);

		// Default lightbox title field
		add_settings_field(
			'default_lightbox_title',
			__( 'Show Title in Lightbox', 'smart-gallery-optimizer' ),
			array( $this, 'render_lightbox_title_field' ),
			$this->page_slug,
			'sgo_link_section'
		);

		// Default lightbox caption field
		add_settings_field(
			'default_lightbox_caption',
			__( 'Show Caption in Lightbox', 'smart-gallery-optimizer' ),
			array( $this, 'render_lightbox_caption_field' ),
			$this->page_slug,
			'sgo_link_section'
		);
	}

	/**
	 * Render settings page
	 */
	public function render_settings_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Include admin page template
		include SGO_PLUGIN_DIR . 'admin/admin-page.php';
	}

	/**
	 * Render conversion section description
	 */
	public function render_conversion_section() {
		echo '<p>' . esc_html__( 'Configure automatic image conversion settings. Images will be converted on upload while preserving the original files.', 'smart-gallery-optimizer' ) . '</p>';

		// Display server capabilities
		$capabilities = Smart_Gallery_Optimizer::get_server_capabilities();
		echo '<div class="sgo-server-info">';
		echo '<h4>' . esc_html__( 'Server Capabilities:', 'smart-gallery-optimizer' ) . '</h4>';
		echo '<ul>';
		echo '<li><span class="dashicons ' . ( $capabilities['gd_library'] ? 'dashicons-yes-alt' : 'dashicons-dismiss' ) . '"></span> ' . esc_html__( 'GD Library', 'smart-gallery-optimizer' ) . '</li>';
		echo '<li><span class="dashicons ' . ( $capabilities['imagick'] ? 'dashicons-yes-alt' : 'dashicons-dismiss' ) . '"></span> ' . esc_html__( 'ImageMagick', 'smart-gallery-optimizer' ) . '</li>';
		echo '<li><span class="dashicons ' . ( $capabilities['webp_support'] ? 'dashicons-yes-alt' : 'dashicons-dismiss' ) . '"></span> ' . esc_html__( 'WebP Support', 'smart-gallery-optimizer' ) . '</li>';
		echo '<li><span class="dashicons ' . ( $capabilities['avif_support'] ? 'dashicons-yes-alt' : 'dashicons-dismiss' ) . '"></span> ' . esc_html__( 'AVIF Support', 'smart-gallery-optimizer' ) . '</li>';
		echo '<li><span class="dashicons ' . ( $capabilities['exif_support'] ? 'dashicons-yes-alt' : 'dashicons-dismiss' ) . '"></span> ' . esc_html__( 'EXIF Support', 'smart-gallery-optimizer' ) . '</li>';
		echo '</ul>';
		echo '</div>';
	}

	/**
	 * Render WebP field
	 */
	public function render_webp_field() {
		$options = get_option( $this->option_name, array() );
		$webp_enabled = isset( $options['webp_conversion'] ) ? $options['webp_conversion'] : '1';
		$capabilities = Smart_Gallery_Optimizer::get_server_capabilities();

		$disabled = ! $capabilities['webp_support'] ? 'disabled' : '';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[webp_conversion]"
				value="1"
				<?php checked( $webp_enabled, '1' ); ?>
				<?php echo esc_attr( $disabled ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php
			if ( ! $capabilities['webp_support'] ) {
				esc_html_e( 'WebP conversion is not supported on this server. GD library with WebP support is required.', 'smart-gallery-optimizer' );
			} else {
				esc_html_e( 'Automatically convert uploaded images to WebP format for better compression and faster loading.', 'smart-gallery-optimizer' );
			}
			?>
		</p>
		<?php
	}

	/**
	 * Render AVIF field
	 */
	public function render_avif_field() {
		$options = get_option( $this->option_name, array() );
		$avif_enabled = isset( $options['avif_conversion'] ) ? $options['avif_conversion'] : '0';
		$capabilities = Smart_Gallery_Optimizer::get_server_capabilities();

		$disabled = ! $capabilities['avif_support'] ? 'disabled' : '';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[avif_conversion]"
				value="1"
				<?php checked( $avif_enabled, '1' ); ?>
				<?php echo esc_attr( $disabled ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php
			if ( ! $capabilities['avif_support'] ) {
				esc_html_e( 'AVIF conversion is not supported on this server. PHP 8.1+ with GD library and AVIF support is required.', 'smart-gallery-optimizer' );
			} else {
				esc_html_e( 'Automatically convert uploaded images to AVIF format for superior compression. Requires PHP 8.1+.', 'smart-gallery-optimizer' );
			}
			?>
		</p>
		<?php
	}

	/**
	 * Render WebP quality field
	 */
	public function render_webp_quality_field() {
		$options = get_option( $this->option_name, array() );
		$quality = isset( $options['webp_quality'] ) ? intval( $options['webp_quality'] ) : 85;
		?>
		<input
			type="number"
			name="<?php echo esc_attr( $this->option_name ); ?>[webp_quality]"
			value="<?php echo esc_attr( $quality ); ?>"
			min="1"
			max="100"
			step="1"
			class="small-text"
		>
		<p class="description">
			<?php esc_html_e( 'Quality level for WebP conversion (1-100). Higher values mean better quality but larger file sizes. Default: 85', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render AVIF quality field
	 */
	public function render_avif_quality_field() {
		$options = get_option( $this->option_name, array() );
		$quality = isset( $options['avif_quality'] ) ? intval( $options['avif_quality'] ) : 85;
		?>
		<input
			type="number"
			name="<?php echo esc_attr( $this->option_name ); ?>[avif_quality]"
			value="<?php echo esc_attr( $quality ); ?>"
			min="1"
			max="100"
			step="1"
			class="small-text"
		>
		<p class="description">
			<?php esc_html_e( 'Quality level for AVIF conversion (1-100). Higher values mean better quality but larger file sizes. Default: 85', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render performance section description
	 */
	public function render_performance_section() {
		echo '<p>' . esc_html__( 'Configure performance and display options for galleries and images.', 'smart-gallery-optimizer' ) . '</p>';
	}

	/**
	 * Render lazy loading field
	 */
	public function render_lazy_loading_field() {
		$options = get_option( $this->option_name, array() );
		$lazy_loading = isset( $options['lazy_loading'] ) ? $options['lazy_loading'] : '1';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[lazy_loading]"
				value="1"
				<?php checked( $lazy_loading, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Load images only when they appear in viewport. Improves page load speed.', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render smart grid field
	 */
	public function render_smart_grid_field() {
		$options = get_option( $this->option_name, array() );
		$smart_grid = isset( $options['smart_grid_enabled'] ) ? $options['smart_grid_enabled'] : '1';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[smart_grid_enabled]"
				value="1"
				<?php checked( $smart_grid, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Use custom responsive grid for [smart_gallery] shortcode. If disabled, WordPress default gallery is used.', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render default columns field
	 */
	public function render_default_columns_field() {
		$options = get_option( $this->option_name, array() );
		$default_columns = isset( $options['default_columns'] ) ? $options['default_columns'] : '3';
		?>
		<select name="<?php echo esc_attr( $this->option_name ); ?>[default_columns]">
			<option value="2" <?php selected( $default_columns, '2' ); ?>>2</option>
			<option value="3" <?php selected( $default_columns, '3' ); ?>>3</option>
			<option value="4" <?php selected( $default_columns, '4' ); ?>>4</option>
			<option value="5" <?php selected( $default_columns, '5' ); ?>>5</option>
			<option value="6" <?php selected( $default_columns, '6' ); ?>>6</option>
		</select>
		<p class="description">
			<?php esc_html_e( 'Default number of columns for galleries (can be overridden in shortcode).', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render EXIF section description
	 */
	public function render_exif_section() {
		echo '<p>' . esc_html__( 'Configure automatic alt-text generation using EXIF data from uploaded images.', 'smart-gallery-optimizer' ) . '</p>';
	}

	/**
	 * Render auto alt-text field
	 */
	public function render_auto_alttext_field() {
		$options = get_option( $this->option_name, array() );
		$enabled = isset( $options['auto_alttext_enabled'] ) ? $options['auto_alttext_enabled'] : '1';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[auto_alttext_enabled]"
				value="1"
				<?php checked( $enabled, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Automatically generate alt-text for images that don\'t have it. Uses EXIF data, title, or filename.', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render alt-text camera field
	 */
	public function render_alttext_camera_field() {
		$options = get_option( $this->option_name, array() );
		$enabled = isset( $options['alttext_include_camera'] ) ? $options['alttext_include_camera'] : '0';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[alttext_include_camera]"
				value="1"
				<?php checked( $enabled, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Include camera make and model in alt-text (e.g., "Canon EOS 5D").', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render alt-text date field
	 */
	public function render_alttext_date_field() {
		$options = get_option( $this->option_name, array() );
		$enabled = isset( $options['alttext_include_date'] ) ? $options['alttext_include_date'] : '0';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[alttext_include_date]"
				value="1"
				<?php checked( $enabled, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Include date taken in alt-text.', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render alt-text location field
	 */
	public function render_alttext_location_field() {
		$options = get_option( $this->option_name, array() );
		$enabled = isset( $options['alttext_include_location'] ) ? $options['alttext_include_location'] : '0';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[alttext_include_location]"
				value="1"
				<?php checked( $enabled, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Include GPS location in alt-text (if available).', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render alt-text template field
	 */
	public function render_alttext_template_field() {
		$options = get_option( $this->option_name, array() );
		$template = isset( $options['alttext_template'] ) ? $options['alttext_template'] : '{description} - Photo taken with {camera} on {date}';
		?>
		<input
			type="text"
			name="<?php echo esc_attr( $this->option_name ); ?>[alttext_template]"
			value="<?php echo esc_attr( $template ); ?>"
			class="regular-text"
		>
		<p class="description">
			<?php esc_html_e( 'Template for alt-text generation. Available variables: {description}, {camera}, {date}, {location}', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render link section description
	 */
	public function render_link_section() {
		echo '<p>' . esc_html__( 'Configure default link behavior for gallery images.', 'smart-gallery-optimizer' ) . '</p>';
	}

	/**
	 * Render link type field
	 */
	public function render_link_type_field() {
		$options = get_option( $this->option_name, array() );
		$link_type = isset( $options['default_link_type'] ) ? $options['default_link_type'] : 'none';
		?>
		<fieldset>
			<label>
				<input type="radio" name="<?php echo esc_attr( $this->option_name ); ?>[default_link_type]" value="none" <?php checked( $link_type, 'none' ); ?>>
				<?php esc_html_e( 'No Link', 'smart-gallery-optimizer' ); ?>
			</label>
			<br>
			<label>
				<input type="radio" name="<?php echo esc_attr( $this->option_name ); ?>[default_link_type]" value="direct" <?php checked( $link_type, 'direct' ); ?>>
				<?php esc_html_e( 'Direct Link to Image File', 'smart-gallery-optimizer' ); ?>
			</label>
			<br>
			<label>
				<input type="radio" name="<?php echo esc_attr( $this->option_name ); ?>[default_link_type]" value="lightbox" <?php checked( $link_type, 'lightbox' ); ?>>
				<?php esc_html_e( 'Lightbox (opens in overlay)', 'smart-gallery-optimizer' ); ?>
			</label>
		</fieldset>
		<p class="description">
			<?php esc_html_e( 'Default link behavior for gallery images (can be overridden per gallery).', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render new tab field
	 */
	public function render_new_tab_field() {
		$options = get_option( $this->option_name, array() );
		$enabled = isset( $options['open_links_new_tab'] ) ? $options['open_links_new_tab'] : '0';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[open_links_new_tab]"
				value="1"
				<?php checked( $enabled, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Open direct image links in a new browser tab.', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render lightbox title field
	 */
	public function render_lightbox_title_field() {
		$options = get_option( $this->option_name, array() );
		$enabled = isset( $options['default_lightbox_title'] ) ? $options['default_lightbox_title'] : '0';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[default_lightbox_title]"
				value="1"
				<?php checked( $enabled, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Display image title in lightbox overlay by default.', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Render lightbox caption field
	 */
	public function render_lightbox_caption_field() {
		$options = get_option( $this->option_name, array() );
		$enabled = isset( $options['default_lightbox_caption'] ) ? $options['default_lightbox_caption'] : '0';
		?>
		<label class="sgo-toggle-switch">
			<input
				type="checkbox"
				name="<?php echo esc_attr( $this->option_name ); ?>[default_lightbox_caption]"
				value="1"
				<?php checked( $enabled, '1' ); ?>
			>
			<span class="sgo-slider"></span>
		</label>
		<p class="description">
			<?php esc_html_e( 'Display image caption in lightbox overlay by default.', 'smart-gallery-optimizer' ); ?>
		</p>
		<?php
	}

	/**
	 * Sanitize settings before saving
	 *
	 * @param array $input Raw input values
	 * @return array Sanitized values
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();

		// Sanitize WebP setting
		$sanitized['webp_conversion'] = isset( $input['webp_conversion'] ) ? '1' : '0';

		// Sanitize AVIF setting (only if supported)
		if ( function_exists( 'imageavif' ) ) {
			$sanitized['avif_conversion'] = isset( $input['avif_conversion'] ) ? '1' : '0';
		} else {
			$sanitized['avif_conversion'] = '0';
		}

		// Sanitize WebP quality (1-100)
		if ( isset( $input['webp_quality'] ) ) {
			$quality = intval( $input['webp_quality'] );
			$sanitized['webp_quality'] = max( 1, min( 100, $quality ) );
		} else {
			$sanitized['webp_quality'] = 85;
		}

		// Sanitize AVIF quality (1-100)
		if ( isset( $input['avif_quality'] ) ) {
			$quality = intval( $input['avif_quality'] );
			$sanitized['avif_quality'] = max( 1, min( 100, $quality ) );
		} else {
			$sanitized['avif_quality'] = 85;
		}

		// Sanitize lazy loading setting
		$sanitized['lazy_loading'] = isset( $input['lazy_loading'] ) ? '1' : '0';

		// Sanitize smart grid setting
		$sanitized['smart_grid_enabled'] = isset( $input['smart_grid_enabled'] ) ? '1' : '0';

		// Sanitize default columns (2-6)
		if ( isset( $input['default_columns'] ) ) {
			$columns = intval( $input['default_columns'] );
			$sanitized['default_columns'] = max( 2, min( 6, $columns ) );
		} else {
			$sanitized['default_columns'] = 3;
		}

		// Sanitize EXIF settings
		$sanitized['auto_alttext_enabled'] = isset( $input['auto_alttext_enabled'] ) ? '1' : '0';
		$sanitized['alttext_include_camera'] = isset( $input['alttext_include_camera'] ) ? '1' : '0';
		$sanitized['alttext_include_date'] = isset( $input['alttext_include_date'] ) ? '1' : '0';
		$sanitized['alttext_include_location'] = isset( $input['alttext_include_location'] ) ? '1' : '0';

		// Sanitize alt-text template
		if ( isset( $input['alttext_template'] ) ) {
			$sanitized['alttext_template'] = sanitize_text_field( $input['alttext_template'] );
		} else {
			$sanitized['alttext_template'] = '{description} - Photo taken with {camera} on {date}';
		}

		// Sanitize link behavior settings
		if ( isset( $input['default_link_type'] ) && in_array( $input['default_link_type'], array( 'none', 'direct', 'lightbox' ), true ) ) {
			$sanitized['default_link_type'] = sanitize_key( $input['default_link_type'] );
		} else {
			$sanitized['default_link_type'] = 'none';
		}

		$sanitized['open_links_new_tab'] = isset( $input['open_links_new_tab'] ) ? '1' : '0';
		$sanitized['default_lightbox_title'] = isset( $input['default_lightbox_title'] ) ? '1' : '0';
		$sanitized['default_lightbox_caption'] = isset( $input['default_lightbox_caption'] ) ? '1' : '0';

		return $sanitized;
	}

	/**
	 * Enqueue admin assets
	 *
	 * @param string $hook Current admin page hook
	 */
	public function enqueue_admin_assets( $hook ) {
		// Only load on our settings page
		if ( 'toplevel_page_' . $this->page_slug !== $hook ) {
			return;
		}

		// Enqueue admin styles
		wp_enqueue_style(
			'sgo-admin-style',
			SGO_PLUGIN_URL . 'admin/css/admin-style.css',
			array(),
			SGO_VERSION
		);
	}
}
