/**
 * PhotoSwipe Initialization - Default Setup with Dynamic Captions
 *
 * Simple PhotoSwipe lightbox initialization with dynamic title/caption support
 *
 * @package Smart_Gallery_Optimizer
 * @since 1.0.0
 */

(function() {
	'use strict';

	// Wait for DOM to be ready
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', initPhotoSwipe);
	} else {
		initPhotoSwipe();
	}

	function initPhotoSwipe() {
		// Check if PhotoSwipe is available
		if (typeof PhotoSwipeLightbox === 'undefined') {
			return;
		}

		// Check if any lightbox elements exist
		var lightboxElements = document.querySelectorAll('.sgo-lightbox');
		if (lightboxElements.length === 0) {
			return;
		}

		// Initialize PhotoSwipe Lightbox with default settings
		var lightbox = new PhotoSwipeLightbox({
			// Gallery and children selectors
			gallery: '.sgo-gallery',
			children: 'a.sgo-lightbox',

			// Core module
			pswpModule: PhotoSwipe,

			// Data source function to properly read image data
			dataSource: function(options) {
				var dataSource = [];
				var galleryElement = options.gallery;
				var linkElements = galleryElement.querySelectorAll('a.sgo-lightbox');

				linkElements.forEach(function(linkEl) {
					var width = parseInt(linkEl.getAttribute('data-pswp-width'), 10);
					var height = parseInt(linkEl.getAttribute('data-pswp-height'), 10);
					var src = linkEl.getAttribute('data-pswp-src') || linkEl.getAttribute('href');

					dataSource.push({
						src: src,
						width: (!isNaN(width) && width > 0) ? width : 1200,
						height: (!isNaN(height) && height > 0) ? height : 900,
						element: linkEl
					});
				});

				return dataSource;
			},

			// Default PhotoSwipe options
			showHideAnimationType: 'zoom',

			// Spacing between slides
			spacing: 0.12,

			// Allow panning to next/prev slide
			allowPanToNext: true,

			// Enable looping
			loop: true,

			// Gestures
			pinchToClose: true,
			closeOnVerticalDrag: true,

			// Keyboard
			escKey: true,
			arrowKeys: true,

			// Mouse wheel to zoom
			wheelToZoom: true,

			// Padding around the image
			padding: { top: 20, bottom: 40, left: 100, right: 100 },

			// Return focus after closing
			returnFocus: true,

			// Initial zoom level
			initialZoomLevel: 'fit',
			secondaryZoomLevel: 1.5,
			maxZoomLevel: 2
		});

		// Add custom caption element with dynamic updates
		lightbox.on('uiRegister', function() {
			lightbox.pswp.ui.registerElement({
				name: 'custom-caption',
				order: 9,
				isButton: false,
				appendTo: 'root',
				onInit: function(el, pswp) {
					// Function to update caption for current slide
					var updateCaption = function() {
						var currSlide = pswp.currSlide;
						var captionHTML = '';

						if (currSlide && currSlide.data && currSlide.data.element) {
							// Get caption from data attribute
							var captionAttr = currSlide.data.element.getAttribute('data-pswp-caption');
							if (captionAttr) {
								captionHTML = captionAttr;
							}
						}

						// Update caption element
						el.innerHTML = captionHTML || '';
					};

					// Update caption on slide change (when navigating between images)
					pswp.on('change', updateCaption);

					// Update caption when content becomes active (first image)
					pswp.on('contentActivate', updateCaption);

					// Update caption immediately after opening
					pswp.on('afterInit', updateCaption);
				}
			});
		});

		// Initialize the lightbox
		lightbox.init();
	}

})();
