/**
 * Smart Gallery Optimizer - Lazy Loading
 *
 * Implements Intersection Observer API for lazy loading images
 * with fallback for older browsers
 *
 * @package Smart_Gallery_Optimizer
 * @version 1.0.0
 */

(function() {
	'use strict';

	/**
	 * Lazy Load Handler Class
	 */
	const SGOLazyLoad = {

		/**
		 * Configuration
		 */
		config: {
			rootMargin: '200px',
			threshold: 0.01,
			fadeInDuration: 300
		},

		/**
		 * Initialize lazy loading
		 */
		init: function() {
			// Check if Intersection Observer is supported
			if ( 'IntersectionObserver' in window ) {
				this.initIntersectionObserver();
			} else {
				// Fallback: Load all images immediately
				this.loadAllImages();
			}
		},

		/**
		 * Initialize Intersection Observer for lazy loading
		 */
		initIntersectionObserver: function() {
			const self = this;

			// Get all lazy load images
			const lazyImages = document.querySelectorAll( 'img.sgo-lazy' );

			if ( lazyImages.length === 0 ) {
				return;
			}

			// Create Intersection Observer
			const imageObserver = new IntersectionObserver( function( entries, observer ) {
				entries.forEach( function( entry ) {
					if ( entry.isIntersecting ) {
						const img = entry.target;
						self.loadImage( img );
						observer.unobserve( img ); // Stop observing after load
					}
				});
			}, {
				rootMargin: self.config.rootMargin,
				threshold: self.config.threshold
			});

			// Observe each lazy image
			lazyImages.forEach( function( img ) {
				imageObserver.observe( img );
			});
		},

		/**
		 * Load individual image
		 *
		 * @param {HTMLImageElement} img Image element to load
		 */
		loadImage: function( img ) {
			const dataSrc = img.getAttribute( 'data-src' );
			const dataSrcset = img.getAttribute( 'data-srcset' );

			if ( ! dataSrc ) {
				return;
			}

			// Create new image to preload
			const tempImage = new Image();

			// Handle successful load
			tempImage.onload = function() {
				// Set actual src and srcset
				img.src = dataSrc;

				if ( dataSrcset ) {
					img.srcset = dataSrcset;
				}

				// Add loaded class for fade-in effect
				img.classList.add( 'sgo-loaded' );
				img.classList.remove( 'sgo-lazy' );

				// Remove data attributes
				img.removeAttribute( 'data-src' );
				img.removeAttribute( 'data-srcset' );
			};

			// Handle load errors
			tempImage.onerror = function() {
				console.warn( 'SGO: Failed to load image:', dataSrc );

				// Still remove lazy class to show alt text
				img.classList.remove( 'sgo-lazy' );
				img.classList.add( 'sgo-error' );

				// Set a fallback or keep placeholder
				if ( img.alt ) {
					img.title = 'Failed to load: ' + img.alt;
				}
			};

			// Start loading
			tempImage.src = dataSrc;

			if ( dataSrcset ) {
				tempImage.srcset = dataSrcset;
			}
		},

		/**
		 * Fallback: Load all images immediately
		 * Used when Intersection Observer is not supported
		 */
		loadAllImages: function() {
			const self = this;
			const lazyImages = document.querySelectorAll( 'img.sgo-lazy' );

			lazyImages.forEach( function( img ) {
				self.loadImage( img );
			});
		}
	};

	/**
	 * Initialize on DOM ready
	 */
	if ( document.readyState === 'loading' ) {
		document.addEventListener( 'DOMContentLoaded', function() {
			SGOLazyLoad.init();
		});
	} else {
		// DOM already loaded
		SGOLazyLoad.init();
	}

	/**
	 * Make available globally for dynamic content
	 */
	window.SGOLazyLoad = SGOLazyLoad;

})();
