/**
 * Gallery Manager JavaScript
 *
 * Handles gallery editor functionality, media library, drag-drop, and AJAX
 *
 * @package Smart_Gallery_Optimizer
 * @since 1.0.0
 */

(function($) {
	'use strict';

	// Gallery List Page functionality
	if ($('.sgo-copy-shortcode').length) {
		// Copy shortcode to clipboard
		$('.sgo-copy-shortcode').on('click', function() {
			var shortcode = $(this).data('shortcode');
			copyToClipboard(shortcode);
		});

		// Delete gallery
		$('.sgo-delete-gallery').on('click', function() {
			if (!confirm(sgoGallery.confirmDelete)) {
				return;
			}

			var $button = $(this);
			var galleryId = $button.data('gallery-id');
			var $row = $button.closest('tr');

			$button.prop('disabled', true).text('Deleting...');

			$.ajax({
				url: sgoGallery.ajaxUrl,
				type: 'POST',
				data: {
					action: 'sgo_delete_gallery',
					nonce: sgoGallery.nonce,
					gallery_id: galleryId
				},
				success: function(response) {
					if (response.success) {
						$row.fadeOut(300, function() {
							$(this).remove();
							// Redirect to show success message
							window.location.href = window.location.pathname + '?page=sgo-gallery-list&deleted=1';
						});
					} else {
						alert(response.data.message || sgoGallery.deleteError);
						$button.prop('disabled', false).text('Delete');
					}
				},
				error: function() {
					alert(sgoGallery.deleteError);
					$button.prop('disabled', false).text('Delete');
				}
			});
		});

		// Duplicate gallery
		$('.sgo-duplicate-gallery').on('click', function() {
			var $button = $(this);
			var galleryId = $button.data('gallery-id');

			$button.prop('disabled', true).text('Duplicating...');

			$.ajax({
				url: sgoGallery.ajaxUrl,
				type: 'POST',
				data: {
					action: 'sgo_duplicate_gallery',
					nonce: sgoGallery.nonce,
					gallery_id: galleryId
				},
				success: function(response) {
					if (response.success) {
						// Redirect to show success message
						window.location.href = window.location.pathname + '?page=sgo-gallery-list&duplicated=1';
					} else {
						alert(response.data.message || 'Failed to duplicate gallery');
						$button.prop('disabled', false).text('Duplicate');
					}
				},
				error: function() {
					alert('Failed to duplicate gallery');
					$button.prop('disabled', false).text('Duplicate');
				}
			});
		});
	}

	// Gallery Editor functionality
	if ($('#sgo-gallery-form').length) {
		var mediaFrame;
		var selectedImages = [];

		// Initialize selected images from existing gallery
		$('.sgo-image-item').each(function() {
			selectedImages.push($(this).data('image-id'));
		});

		// Initialize jQuery UI Sortable for drag and drop
		$('#sgo-selected-images').sortable({
			placeholder: 'ui-sortable-placeholder',
			cursor: 'move',
			tolerance: 'pointer',
			update: function() {
				updateSelectedImages();
			}
		});

		// Add images button
		$('#sgo-add-images').on('click', function(e) {
			e.preventDefault();

			// If the media frame already exists, reopen it
			if (mediaFrame) {
				mediaFrame.open();
				return;
			}

			// Create a new media frame
			mediaFrame = wp.media({
				title: 'Select Gallery Images',
				button: {
					text: 'Add to Gallery'
				},
				multiple: true
			});

			// When images are selected
			mediaFrame.on('select', function() {
				var selection = mediaFrame.state().get('selection');

				selection.each(function(attachment) {
					attachment = attachment.toJSON();

					// Check if image is already selected
					if (selectedImages.indexOf(attachment.id) === -1) {
						selectedImages.push(attachment.id);
						addImageToGallery(attachment);
					}
				});

				updateImageCount();
			});

			// Open the media frame
			mediaFrame.open();
		});

		// Remove image
		$(document).on('click', '.sgo-remove-image', function() {
			var $item = $(this).closest('.sgo-image-item');
			var imageId = $item.data('image-id');

			// Remove from array
			var index = selectedImages.indexOf(imageId);
			if (index > -1) {
				selectedImages.splice(index, 1);
			}

			// Remove from DOM
			$item.fadeOut(200, function() {
				$(this).remove();
				updateImageCount();
			});
		});

		// Add image to gallery display
		function addImageToGallery(attachment) {
			var thumbnail = attachment.sizes && attachment.sizes.thumbnail
				? attachment.sizes.thumbnail.url
				: attachment.url;

			var imageHtml =
				'<div class="sgo-image-item" data-image-id="' + attachment.id + '">' +
					'<img src="' + thumbnail + '" alt="' + (attachment.alt || attachment.title) + '">' +
					'<button type="button" class="sgo-remove-image" title="Remove image">' +
						'<span class="dashicons dashicons-no-alt"></span>' +
					'</button>' +
					'<div class="sgo-image-drag-handle">' +
						'<span class="dashicons dashicons-move"></span>' +
					'</div>' +
				'</div>';

			$('#sgo-selected-images').append(imageHtml);
		}

		// Update selected images array from DOM
		function updateSelectedImages() {
			selectedImages = [];
			$('.sgo-image-item').each(function() {
				selectedImages.push($(this).data('image-id'));
			});
		}

		// Update image count display
		function updateImageCount() {
			$('#sgo-image-count-value').text(selectedImages.length);
		}

		// Link type change handler
		$('input[name="link_type"]').on('change', function() {
			var linkType = $(this).val();

			if (linkType === 'lightbox') {
				$('#sgo-lightbox-options').slideDown(200);
			} else {
				$('#sgo-lightbox-options').slideUp(200);
			}

			updateGeneratedShortcode();
		});

		// Update generated shortcode
		function updateGeneratedShortcode() {
			var galleryId = $('#sgo-gallery-id').val();

			if (galleryId) {
				var shortcode = '[smart_gallery id="' + galleryId + '"]';
				$('#sgo-generated-shortcode').text(shortcode);
				$('#sgo-copy-shortcode-btn').data('shortcode', shortcode);
			}
		}

		// Copy shortcode button
		$('#sgo-copy-shortcode-btn').on('click', function() {
			var shortcode = $(this).data('shortcode');
			copyToClipboard(shortcode);
		});

		// Save gallery form
		$('#sgo-gallery-form').on('submit', function(e) {
			e.preventDefault();

			var $form = $(this);
			var $saveButton = $('#sgo-save-gallery');
			var $status = $('#sgo-save-status');

			// Validate gallery name
			var galleryName = $('#gallery-name').val().trim();
			if (!galleryName) {
				alert('Please enter a gallery name');
				$('#gallery-name').focus();
				return;
			}

			// Update selected images from DOM order
			updateSelectedImages();

			// Prepare form data
			var formData = {
				action: 'sgo_save_gallery',
				nonce: sgoGallery.nonce,
				gallery_id: $('#sgo-gallery-id').val(),
				gallery_name: galleryName,
				image_ids: selectedImages,
				columns: $('#gallery-columns').val(),
				image_size: $('#gallery-size').val(),
				gap: $('#gallery-gap').val(),
				link_type: $('input[name="link_type"]:checked').val(),
				lightbox_title: $('input[name="lightbox_title"]').is(':checked') ? 1 : 0,
				lightbox_caption: $('input[name="lightbox_caption"]').is(':checked') ? 1 : 0
			};

			// Show loading state
			$form.addClass('loading');
			$saveButton.prop('disabled', true);
			$status.removeClass('success error').addClass('loading').text('Saving...');

			// Send AJAX request
			$.ajax({
				url: sgoGallery.ajaxUrl,
				type: 'POST',
				data: formData,
				success: function(response) {
					if (response.success) {
						$status.removeClass('loading').addClass('success').text(sgoGallery.saveSuccess);

						// Update gallery ID if it's a new gallery
						if (response.data.gallery_id) {
							$('#sgo-gallery-id').val(response.data.gallery_id);

							// Update shortcode
							var shortcode = response.data.shortcode;
							$('#sgo-generated-shortcode').text(shortcode);
							$('#sgo-copy-shortcode-btn').data('shortcode', shortcode).show();

							// Update URL without reload
							var newUrl = window.location.pathname + '?page=sgo-gallery-editor&gallery_id=' + response.data.gallery_id;
							window.history.pushState({}, '', newUrl);
						}

						// Redirect to gallery list after 1.5 seconds with success message
						setTimeout(function() {
							window.location.href = window.location.pathname + '?page=sgo-gallery-list&saved=1';
						}, 1500);
					} else {
						$status.removeClass('loading').addClass('error').text(response.data.message || sgoGallery.saveError);
					}
				},
				error: function() {
					$status.removeClass('loading').addClass('error').text(sgoGallery.saveError);
				},
				complete: function() {
					$form.removeClass('loading');
					$saveButton.prop('disabled', false);
				}
			});
		});
	}

	// Copy to clipboard utility function
	function copyToClipboard(text) {
		if (navigator.clipboard && navigator.clipboard.writeText) {
			navigator.clipboard.writeText(text).then(function() {
				showCopyNotification(true);
			}).catch(function() {
				fallbackCopyToClipboard(text);
			});
		} else {
			fallbackCopyToClipboard(text);
		}
	}

	// Fallback copy method for older browsers
	function fallbackCopyToClipboard(text) {
		var textArea = document.createElement('textarea');
		textArea.value = text;
		textArea.style.position = 'fixed';
		textArea.style.top = '-9999px';
		textArea.style.left = '-9999px';
		document.body.appendChild(textArea);
		textArea.focus();
		textArea.select();

		try {
			var successful = document.execCommand('copy');
			showCopyNotification(successful);
		} catch (err) {
			showCopyNotification(false);
		}

		document.body.removeChild(textArea);
	}

	// Show copy notification
	function showCopyNotification(success) {
		var message = success ? sgoGallery.copySuccess : sgoGallery.copyError;

		// Create notification element
		var $notification = $('<div class="notice notice-' + (success ? 'success' : 'error') + ' is-dismissible" style="position: fixed; top: 32px; right: 20px; z-index: 9999; max-width: 300px;"><p>' + message + '</p></div>');

		$('body').append($notification);

		// Auto-hide after 3 seconds
		setTimeout(function() {
			$notification.fadeOut(300, function() {
				$(this).remove();
			});
		}, 3000);
	}

})(jQuery);
