<?php
/**
 * Gallery List Admin Page
 *
 * Displays all saved galleries in a table format
 *
 * @package Smart_Gallery_Optimizer
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get current page and search query
// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Search and pagination do not require nonces
$paged = isset( $_GET['paged'] ) ? absint( $_GET['paged'] ) : 1;
// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Search and pagination do not require nonces
$search = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';

// Build query args
$args = array(
	'paged' => $paged,
);

if ( ! empty( $search ) ) {
	$args['s'] = $search;
}

// Get galleries
$result = SGO_Gallery_Manager::get_all_galleries( $args );
$galleries = $result['galleries'];
$total_galleries = $result['total'];
$total_pages = $result['total_pages'];

?>

<div class="wrap">
	<h1 class="wp-heading-inline">
		<?php esc_html_e( 'Gallery List', 'smart-gallery-optimizer' ); ?>
	</h1>

	<a href="<?php echo esc_url( admin_url( 'admin.php?page=sgo-gallery-editor' ) ); ?>" class="page-title-action">
		<?php esc_html_e( 'Create New Gallery', 'smart-gallery-optimizer' ); ?>
	</a>

	<hr class="wp-header-end">

	<?php
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just displaying a success message
	if ( isset( $_GET['saved'] ) && '1' === $_GET['saved'] ) :
		?>

		<div class="notice notice-success is-dismissible">
			<p><?php esc_html_e( 'Gallery saved successfully!', 'smart-gallery-optimizer' ); ?></p>
		</div>
	<?php endif; ?>

	<?php
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just displaying a success message
	if ( isset( $_GET['deleted'] ) && '1' === $_GET['deleted'] ) :
		?>

		<div class="notice notice-success is-dismissible">
			<p><?php esc_html_e( 'Gallery deleted successfully!', 'smart-gallery-optimizer' ); ?></p>
		</div>
	<?php endif; ?>

	<?php
	// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Just displaying a success message
	if ( isset( $_GET['duplicated'] ) && '1' === $_GET['duplicated'] ) :
		?>

		<div class="notice notice-success is-dismissible">
			<p><?php esc_html_e( 'Gallery duplicated successfully!', 'smart-gallery-optimizer' ); ?></p>
		</div>
	<?php endif; ?>

	<!-- Search form -->
	<form method="get" class="sgo-search-form" style="margin: 20px 0;">
		<input type="hidden" name="page" value="sgo-gallery-list">
		<p class="search-box">
			<label class="screen-reader-text" for="gallery-search-input">
				<?php esc_html_e( 'Search galleries:', 'smart-gallery-optimizer' ); ?>
			</label>
			<input type="search" id="gallery-search-input" name="s" value="<?php echo esc_attr( $search ); ?>" placeholder="<?php esc_attr_e( 'Search galleries...', 'smart-gallery-optimizer' ); ?>">
			<input type="submit" id="search-submit" class="button" value="<?php esc_attr_e( 'Search Galleries', 'smart-gallery-optimizer' ); ?>">
		</p>
	</form>

	<!-- Galleries table -->
	<?php if ( ! empty( $galleries ) ) : ?>
		<table class="wp-list-table widefat fixed striped">
			<thead>
				<tr>
					<th scope="col" class="column-name column-primary">
						<?php esc_html_e( 'Gallery Name', 'smart-gallery-optimizer' ); ?>
					</th>
					<th scope="col" class="column-shortcode">
						<?php esc_html_e( 'Shortcode', 'smart-gallery-optimizer' ); ?>
					</th>
					<th scope="col" class="column-images">
						<?php esc_html_e( 'Images', 'smart-gallery-optimizer' ); ?>
					</th>
					<th scope="col" class="column-settings">
						<?php esc_html_e( 'Settings', 'smart-gallery-optimizer' ); ?>
					</th>
					<th scope="col" class="column-actions">
						<?php esc_html_e( 'Actions', 'smart-gallery-optimizer' ); ?>
					</th>
				</tr>
			</thead>
			<tbody>
				<?php foreach ( $galleries as $gallery ) : ?>
					<tr data-gallery-id="<?php echo esc_attr( $gallery['id'] ); ?>">
						<td class="column-name column-primary" data-colname="<?php esc_attr_e( 'Gallery Name', 'smart-gallery-optimizer' ); ?>">
							<strong>
								<a href="<?php echo esc_url( admin_url( 'admin.php?page=sgo-gallery-editor&gallery_id=' . $gallery['id'] ) ); ?>">
									<?php echo esc_html( $gallery['name'] ); ?>
								</a>
							</strong>
							<button type="button" class="toggle-row">
								<span class="screen-reader-text"><?php esc_html_e( 'Show more details', 'smart-gallery-optimizer' ); ?></span>
							</button>
						</td>
						<td class="column-shortcode" data-colname="<?php esc_attr_e( 'Shortcode', 'smart-gallery-optimizer' ); ?>">
							<div class="shortcode-wrapper">
								<code class="sgo-shortcode"><?php echo esc_html( $gallery['shortcode'] ); ?></code>
								<button type="button" class="button button-small sgo-copy-shortcode" data-shortcode="<?php echo esc_attr( $gallery['shortcode'] ); ?>" title="<?php esc_attr_e( 'Copy shortcode', 'smart-gallery-optimizer' ); ?>">
									<span class="dashicons dashicons-clipboard"></span>
								</button>
							</div>
						</td>
						<td class="column-images" data-colname="<?php esc_attr_e( 'Images', 'smart-gallery-optimizer' ); ?>">
							<span class="image-count">
								<?php
								printf(
									/* translators: %d: number of images */
									esc_html( _n( '%d image', '%d images', $gallery['image_count'], 'smart-gallery-optimizer' ) ),
									absint( $gallery['image_count'] )
								);
								?>
							</span>
						</td>
						<td class="column-settings" data-colname="<?php esc_attr_e( 'Settings', 'smart-gallery-optimizer' ); ?>">
							<?php
							$settings_text = sprintf(
								/* translators: 1: number of columns, 2: image size, 3: link type */
								esc_html__( '%1$d cols, %2$s, %3$s', 'smart-gallery-optimizer' ),
								absint( $gallery['columns'] ),
								esc_html( $gallery['size'] ),
								esc_html( $gallery['link_type'] )
							);
							echo esc_html( $settings_text );
							?>
						</td>
						<td class="column-actions" data-colname="<?php esc_attr_e( 'Actions', 'smart-gallery-optimizer' ); ?>">
							<a href="<?php echo esc_url( admin_url( 'admin.php?page=sgo-gallery-editor&gallery_id=' . $gallery['id'] ) ); ?>" class="button button-small">
								<?php esc_html_e( 'Edit', 'smart-gallery-optimizer' ); ?>
							</a>
							<button type="button" class="button button-small sgo-duplicate-gallery" data-gallery-id="<?php echo esc_attr( $gallery['id'] ); ?>">
								<?php esc_html_e( 'Duplicate', 'smart-gallery-optimizer' ); ?>
							</button>
							<button type="button" class="button button-small button-link-delete sgo-delete-gallery" data-gallery-id="<?php echo esc_attr( $gallery['id'] ); ?>">
								<?php esc_html_e( 'Delete', 'smart-gallery-optimizer' ); ?>
							</button>
						</td>
					</tr>
				<?php endforeach; ?>
			</tbody>
			<tfoot>
				<tr>
					<th scope="col" class="column-name column-primary">
						<?php esc_html_e( 'Gallery Name', 'smart-gallery-optimizer' ); ?>
					</th>
					<th scope="col" class="column-shortcode">
						<?php esc_html_e( 'Shortcode', 'smart-gallery-optimizer' ); ?>
					</th>
					<th scope="col" class="column-images">
						<?php esc_html_e( 'Images', 'smart-gallery-optimizer' ); ?>
					</th>
					<th scope="col" class="column-settings">
						<?php esc_html_e( 'Settings', 'smart-gallery-optimizer' ); ?>
					</th>
					<th scope="col" class="column-actions">
						<?php esc_html_e( 'Actions', 'smart-gallery-optimizer' ); ?>
					</th>
				</tr>
			</tfoot>
		</table>

		<!-- Pagination -->
		<?php if ( $total_pages > 1 ) : ?>
			<div class="tablenav bottom">
				<div class="tablenav-pages">
					<?php
					$page_links = paginate_links(
						array(
							'base'      => add_query_arg( 'paged', '%#%' ),
							'format'    => '',
							'prev_text' => __( '&laquo;', 'smart-gallery-optimizer' ),
							'next_text' => __( '&raquo;', 'smart-gallery-optimizer' ),
							'total'     => $total_pages,
							'current'   => $paged,
						)
					);

					if ( $page_links ) {
						echo '<span class="displaying-num">' .
							sprintf(
								/* translators: %d: total number of galleries */
								esc_html( _n( '%d gallery', '%d galleries', $total_galleries, 'smart-gallery-optimizer' ) ),
								absint( $total_galleries )
							) .
							'</span>';
						echo '<span class="pagination-links">' . wp_kses_post( $page_links ) . '</span>';
					}
					?>
				</div>
			</div>
		<?php endif; ?>

	<?php else : ?>
		<div class="notice notice-info" style="margin-top: 20px;">
			<p>
				<?php if ( ! empty( $search ) ) : ?>
					<?php esc_html_e( 'No galleries found matching your search.', 'smart-gallery-optimizer' ); ?>
				<?php else : ?>
					<?php esc_html_e( 'No galleries found. Create your first gallery to get started!', 'smart-gallery-optimizer' ); ?>
				<?php endif; ?>
			</p>
			<?php if ( empty( $search ) ) : ?>
				<p>
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=sgo-gallery-editor' ) ); ?>" class="button button-primary">
						<?php esc_html_e( 'Create Your First Gallery', 'smart-gallery-optimizer' ); ?>
					</a>
				</p>
			<?php endif; ?>
		</div>
	<?php endif; ?>

</div>
