<?php
/**
 * Gallery Editor Admin Page
 *
 * Create and edit galleries with image selection and settings
 *
 * @package Smart_Gallery_Optimizer
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Get gallery ID if editing
$gallery_id = isset( $_GET['gallery_id'] ) ? absint( $_GET['gallery_id'] ) : 0;
$is_editing = $gallery_id > 0;

// Get gallery data if editing
$gallery_data = null;
if ( $is_editing ) {
	$gallery_data = SGO_Gallery_Manager::get_gallery_data( $gallery_id );
	if ( ! $gallery_data ) {
		wp_die( esc_html__( 'Gallery not found.', 'smart-gallery-optimizer' ) );
	}
}

// Set default values
$gallery_name = $is_editing ? $gallery_data['name'] : '';
$gallery_images = $is_editing && is_array( $gallery_data['images'] ) ? $gallery_data['images'] : array();
$gallery_columns = $is_editing && $gallery_data['columns'] ? $gallery_data['columns'] : 3;
$gallery_size = $is_editing && $gallery_data['size'] ? $gallery_data['size'] : 'medium';
$gallery_gap = $is_editing && $gallery_data['gap'] ? $gallery_data['gap'] : 20;
$gallery_link_type = $is_editing && $gallery_data['link_type'] ? $gallery_data['link_type'] : 'none';
$gallery_lightbox_title = $is_editing ? $gallery_data['lightbox_title'] : false;
$gallery_lightbox_caption = $is_editing ? $gallery_data['lightbox_caption'] : false;

?>

<div class="wrap sgo-gallery-editor">
	<h1>
		<?php echo $is_editing ? esc_html__( 'Edit Gallery', 'smart-gallery-optimizer' ) : esc_html__( 'Create New Gallery', 'smart-gallery-optimizer' ); ?>
	</h1>

	<form id="sgo-gallery-form" class="sgo-gallery-form">
		<?php wp_nonce_field( 'sgo_gallery_nonce', 'sgo_gallery_nonce' ); ?>
		<input type="hidden" id="sgo-gallery-id" name="gallery_id" value="<?php echo esc_attr( $gallery_id ); ?>">

		<div class="sgo-form-section">
			<h2><?php esc_html_e( 'Basic Information', 'smart-gallery-optimizer' ); ?></h2>
			<table class="form-table" role="presentation">
				<tbody>
					<tr>
						<th scope="row">
							<label for="gallery-name">
								<?php esc_html_e( 'Gallery Name', 'smart-gallery-optimizer' ); ?>
								<span class="required">*</span>
							</label>
						</th>
						<td>
							<input type="text" id="gallery-name" name="gallery_name" class="regular-text" value="<?php echo esc_attr( $gallery_name ); ?>" required>
							<p class="description">
								<?php esc_html_e( 'Enter a descriptive name for this gallery.', 'smart-gallery-optimizer' ); ?>
							</p>
						</td>
					</tr>
				</tbody>
			</table>
		</div>

		<div class="sgo-form-section">
			<h2><?php esc_html_e( 'Image Selection', 'smart-gallery-optimizer' ); ?></h2>
			<div class="sgo-image-selection">
				<button type="button" id="sgo-add-images" class="button button-primary">
					<span class="dashicons dashicons-images-alt2"></span>
					<?php esc_html_e( 'Add Images', 'smart-gallery-optimizer' ); ?>
				</button>
				<p class="description">
					<?php esc_html_e( 'Click to select images from your media library. You can drag and drop to reorder.', 'smart-gallery-optimizer' ); ?>
				</p>

				<div id="sgo-selected-images" class="sgo-selected-images">
					<?php if ( ! empty( $gallery_images ) ) : ?>
						<?php foreach ( $gallery_images as $image_id ) : ?>
							<?php
							$image_url = wp_get_attachment_image_url( $image_id, 'thumbnail' );
							$image_title = get_the_title( $image_id );
							if ( $image_url ) :
							?>
								<div class="sgo-image-item" data-image-id="<?php echo esc_attr( $image_id ); ?>">
									<img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( $image_title ); ?>">
									<button type="button" class="sgo-remove-image" title="<?php esc_attr_e( 'Remove image', 'smart-gallery-optimizer' ); ?>">
										<span class="dashicons dashicons-no-alt"></span>
									</button>
									<div class="sgo-image-drag-handle">
										<span class="dashicons dashicons-move"></span>
									</div>
								</div>
							<?php endif; ?>
						<?php endforeach; ?>
					<?php endif; ?>
				</div>

				<p class="sgo-image-count">
					<strong><?php esc_html_e( 'Images:', 'smart-gallery-optimizer' ); ?></strong>
					<span id="sgo-image-count-value"><?php echo count( $gallery_images ); ?></span>
				</p>
			</div>
		</div>

		<div class="sgo-form-section">
			<h2><?php esc_html_e( 'Display Settings', 'smart-gallery-optimizer' ); ?></h2>
			<table class="form-table" role="presentation">
				<tbody>
					<tr>
						<th scope="row">
							<label for="gallery-columns">
								<?php esc_html_e( 'Columns', 'smart-gallery-optimizer' ); ?>
							</label>
						</th>
						<td>
							<select id="gallery-columns" name="columns">
								<?php for ( $i = 2; $i <= 6; $i++ ) : ?>
									<option value="<?php echo esc_attr( $i ); ?>" <?php selected( $gallery_columns, $i ); ?>>
										<?php echo esc_html( $i ); ?>
									</option>
								<?php endfor; ?>
							</select>
							<p class="description">
								<?php esc_html_e( 'Number of columns in the gallery grid.', 'smart-gallery-optimizer' ); ?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="gallery-size">
								<?php esc_html_e( 'Image Size', 'smart-gallery-optimizer' ); ?>
							</label>
						</th>
						<td>
							<select id="gallery-size" name="image_size">
								<option value="thumbnail" <?php selected( $gallery_size, 'thumbnail' ); ?>>
									<?php esc_html_e( 'Thumbnail', 'smart-gallery-optimizer' ); ?>
								</option>
								<option value="medium" <?php selected( $gallery_size, 'medium' ); ?>>
									<?php esc_html_e( 'Medium', 'smart-gallery-optimizer' ); ?>
								</option>
								<option value="large" <?php selected( $gallery_size, 'large' ); ?>>
									<?php esc_html_e( 'Large', 'smart-gallery-optimizer' ); ?>
								</option>
								<option value="full" <?php selected( $gallery_size, 'full' ); ?>>
									<?php esc_html_e( 'Full Size', 'smart-gallery-optimizer' ); ?>
								</option>
							</select>
							<p class="description">
								<?php esc_html_e( 'Select the image size to display in the gallery.', 'smart-gallery-optimizer' ); ?>
							</p>
						</td>
					</tr>
					<tr>
						<th scope="row">
							<label for="gallery-gap">
								<?php esc_html_e( 'Gap Spacing', 'smart-gallery-optimizer' ); ?>
							</label>
						</th>
						<td>
							<input type="number" id="gallery-gap" name="gap" min="0" max="100" value="<?php echo esc_attr( $gallery_gap ); ?>" class="small-text">
							<span class="description">px</span>
							<p class="description">
								<?php esc_html_e( 'Space between images in pixels (0-100).', 'smart-gallery-optimizer' ); ?>
							</p>
						</td>
					</tr>
				</tbody>
			</table>
		</div>

		<div class="sgo-form-section">
			<h2><?php esc_html_e( 'Link Behavior', 'smart-gallery-optimizer' ); ?></h2>
			<table class="form-table" role="presentation">
				<tbody>
					<tr>
						<th scope="row">
							<?php esc_html_e( 'Link Type', 'smart-gallery-optimizer' ); ?>
						</th>
						<td>
							<fieldset>
								<label>
									<input type="radio" name="link_type" value="none" <?php checked( $gallery_link_type, 'none' ); ?>>
									<?php esc_html_e( 'No Link', 'smart-gallery-optimizer' ); ?>
								</label>
								<br>
								<label>
									<input type="radio" name="link_type" value="direct" <?php checked( $gallery_link_type, 'direct' ); ?>>
									<?php esc_html_e( 'Direct Link to Image File', 'smart-gallery-optimizer' ); ?>
								</label>
								<br>
								<label>
									<input type="radio" name="link_type" value="lightbox" <?php checked( $gallery_link_type, 'lightbox' ); ?>>
									<?php esc_html_e( 'Lightbox (opens in overlay)', 'smart-gallery-optimizer' ); ?>
								</label>
							</fieldset>
						</td>
					</tr>
					<tr id="sgo-lightbox-options" style="<?php echo ( $gallery_link_type !== 'lightbox' ) ? 'display: none;' : ''; ?>">
						<th scope="row">
							<?php esc_html_e( 'Lightbox Options', 'smart-gallery-optimizer' ); ?>
						</th>
						<td>
							<fieldset>
								<label>
									<input type="checkbox" name="lightbox_title" value="1" <?php checked( $gallery_lightbox_title, true ); ?>>
									<?php esc_html_e( 'Show image title in lightbox', 'smart-gallery-optimizer' ); ?>
								</label>
								<br>
								<label>
									<input type="checkbox" name="lightbox_caption" value="1" <?php checked( $gallery_lightbox_caption, true ); ?>>
									<?php esc_html_e( 'Show image caption in lightbox', 'smart-gallery-optimizer' ); ?>
								</label>
							</fieldset>
						</td>
					</tr>
				</tbody>
			</table>
		</div>

		<div class="sgo-form-section sgo-shortcode-section">
			<h2><?php esc_html_e( 'Generated Shortcode', 'smart-gallery-optimizer' ); ?></h2>
			<div class="sgo-shortcode-box">
				<code id="sgo-generated-shortcode">
					<?php
					if ( $is_editing ) {
						echo esc_html( '[smart_gallery id="' . $gallery_id . '"]' );
					} else {
						echo esc_html__( 'Save gallery to generate shortcode', 'smart-gallery-optimizer' );
					}
					?>
				</code>
				<?php if ( $is_editing ) : ?>
					<button type="button" id="sgo-copy-shortcode-btn" class="button button-secondary" data-shortcode="<?php echo esc_attr( '[smart_gallery id="' . $gallery_id . '"]' ); ?>">
						<span class="dashicons dashicons-clipboard"></span>
						<?php esc_html_e( 'Copy Shortcode', 'smart-gallery-optimizer' ); ?>
					</button>
				<?php endif; ?>
			</div>
			<p class="description">
				<?php esc_html_e( 'Copy this shortcode and paste it into any post or page to display the gallery.', 'smart-gallery-optimizer' ); ?>
			</p>
		</div>

		<p class="submit">
			<button type="submit" id="sgo-save-gallery" class="button button-primary button-large">
				<?php esc_html_e( 'Save Gallery', 'smart-gallery-optimizer' ); ?>
			</button>
			<a href="<?php echo esc_url( admin_url( 'admin.php?page=sgo-gallery-list' ) ); ?>" class="button button-secondary button-large">
				<?php esc_html_e( 'Cancel', 'smart-gallery-optimizer' ); ?>
			</a>
			<span id="sgo-save-status" class="sgo-save-status"></span>
		</p>
	</form>
</div>
